/**************************************************************************
 *   Copyright (C) 2005 by Oleksandr Shneyder                              *
 *   oleksandr.shneyder@treuchtlingen.de                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "aktenplan.h"
#include "alistview.h"
#include "atabview.h"
#include <QTreeWidgetItem>

#include <QToolButton>
#include <QTimer>
#include <QAction>
#include <QMimeData>
#include <QApplication>
#include <QToolBar>
#include <QTreeWidgetItemIterator>
#include <QFileDialog>
#include <QContextMenuEvent>
//#include <kaboutkde.h>
// #include <kapp.h>
//#include <kdirselectdialog.h>
//#include <kfileitem.h>
//#include <kfiledialog.h>
//#include <kiconloader.h>
#include <QLineEdit>
// #include <klocale.h>
#include <QMainWindow>
#include <QMenuBar>
#include <QMessageBox>
// #include <knotifyclient.h>
// #include <kio/netaccess.h>
#include <QMenu>
#include <QProcess>
#include <QPushButton>
#include <QSettings>
#include <QStatusBar>
// #include <kstdaccel.h>
// #include <kstdaction.h>


#include <QClipboard>
#include <QDrag>
#include <QFile>
#include <QLabel>
// #include <QTreeWidget>
#include <QWhatsThis>
#include <QDebug>

aKtenplan::aKtenplan()
{
    tableView=false;
    mConfig=new QSettings("Stadt Treuchtlingen", "Aktenplan");
    fileName=mConfig->value("fname",QVariant("")).toString();
    rootURL=mConfig->value("root",QVariant("")).toString();
    if(fileName.length()<=0)
        slotGetFname();

    setWindowIcon(QIcon::fromTheme("fileplan-app"));
    qDebug()<<"Filename: "<<fileName;
    searchChanged=false;
    setWindowTitle("aKtenplan");
    QString msg="0";
    msg+=" Suchergebnisse";
    mStatus=new QLabel(msg,statusBar());
    statusBar()->addWidget(mStatus);

    QMenu *fileMenu=menuBar()->addMenu(tr("&File"));

    QMenu *editMenu=menuBar()->addMenu(tr("&Edit"));

    QMenu *viewMenu=menuBar()->addMenu(tr("&View"));

    QMenu *help=menuBar()->addMenu(tr("&Help"));

    help->addAction(QWhatsThis::createAction(this));

    help->addAction(QIcon::fromTheme("assistant"), tr("About &Qt"), this, SLOT(slotAboutQt()));
    help->addAction(QIcon::fromTheme("fileplan-app"), tr("About aKtenplan"), this, SLOT(slotAbout()));

    QToolBar* tbFile=addToolBar("File");
    QToolBar* tbEdit=addToolBar("Edit");
    QToolBar* tbView=addToolBar("View");
    QToolBar* tbSearch=addToolBar("Search");


    QAction* open=fileMenu->addAction(style()->standardIcon(QStyle::SP_FileIcon),
                                      tr("Open"),this, SLOT(slotFOpen()), QKeySequence(QKeySequence::Open));
    open->setToolTip(tr("Datei oeffnen\nTipp:\nNutzen Sie \"Strg\" + \"o\"\num Dateien zu oeffnen"));


    tbFile->addAction(open);

    tbFile->addAction( fileMenu->addAction(style()->standardIcon(QStyle::SP_DirIcon),
                                           tr("Select File Plan Root"),this, SLOT(slotSelectRoot())));

    fileMenu->addAction(style()->standardIcon(QStyle::SP_DirIcon),
                                           tr("Create File Plan Structure"),this, SLOT(slotCreateFolder()));



    fileMenu->addSeparator();


    fileMenu->addAction(QIcon::fromTheme("application-exit"),
                        tr("Quit"),QCoreApplication::instance(), SLOT(quit()));






    copy=editMenu->addAction(QIcon::fromTheme("edit-copy"),tr("Copy"),this, SLOT(slotCopy()),QKeySequence(QKeySequence::Copy));

    copy->setToolTip(tr("Suchergebnisse kopieren\nTipp:\nNutzen Sie \"strg\" + \"c\"\num Suchergebnisse zu kopieren.\n Suchergebnisse lassen sich auch per Drag&Drop einfuegen!"));

    tbEdit->addAction(copy);

    copyAll=editMenu->addAction(QIcon::fromTheme("edit-select-all"),tr("&Alle Suchergebnisse kopieren"),this, SLOT(slotCopyAll()),QKeySequence(QKeySequence::SelectAll));

    copyAll->setToolTip(tr("Alle Suchergebnisse kopieren\nTipp:\nNutzen Sie \"strg\" + \"a\"\num Suchergebnisse zu kopieren.\n Suchergebnisse lassen sich auch per Drag&Drop einfuegen!"));

    tbEdit->addAction(copyAll);


    openFolder=new QAction(style()->standardIcon(QStyle::SP_DirIcon), tr("Open folder"));

    connect(openFolder, SIGNAL(triggered()),this, SLOT(slotOpenFolder()));




    pSwitch=viewMenu->addAction(QIcon::fromTheme("table"),tr("&Tableview"),this,SLOT(slotSwitchView()), QKeySequence(Qt::CTRL + Qt::Key_T));

    tbView->addAction(pSwitch);



    QLabel* label = new QLabel(tr("&Begriff:"),tbSearch);
    lineEdit = new QLineEdit(tbSearch);
    label->setBuddy(lineEdit);

    lineEdit->setToolTip(tr("Tipp:\n Wenn Sie nicht den gesamten Begriff kennen:\n aKtenplan sucht auch unvollstaendige Begriffe"));


    findButton = new QToolButton(tbSearch);
    findButton->setIcon(QIcon::fromTheme("edit-find"));
    findButton->setEnabled(false);
    findButton->setToolTip("Suchen");

    tbSearch->addWidget(label);
    tbSearch->addWidget(lineEdit);
    tbSearch->addWidget(findButton);

    connect(lineEdit,SIGNAL(textChanged(const QString&)),this,
            SLOT(slotSearchText(const QString& )));

    connect(findButton,SIGNAL(clicked()),this,
            SLOT(slotFind()));


    initList();
    //setAutoSaveSettings ();
    setAcceptDrops(true);
}

aKtenplan::~aKtenplan()
{}

void aKtenplan::initList()
{
    pSwitch->setWhatsThis(tr("Tipp:\n Die Tabellenansicht zeigt alle Aktenzeichen uebersichtlich in einer Tabelle an."));
    pSwitch->setToolTip(tr("Tableview"));
    pSwitch->setText(tr("&Tableview"));
    pSwitch->setIcon(QIcon::fromTheme("table"));
    numOfRaws=0;
    list=new AListView(this);
    setCentralWidget(list);
    list->show();
    list->setColumnCount(6);
    QStringList labels;
    labels<< tr("Aktenzeichen")<<tr("Beschreibung")<<tr("ABFK")<<tr("ABFL")<<tr("Bemerkungen")<<tr("Fundstellen");
    list->setHeaderLabels(labels);
    list->setRootIsDecorated(true);
    list->setSelectionMode(QAbstractItemView::ExtendedSelection);
    list->setAllColumnsShowFocus(true);
    //     list->setDragEnabled(true);
    //     list->setDragDropMode(QAbstractItemView::DragOnly);
    //     list->setDropIndicatorShown(false);
    lastItem=0;
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QMessageBox::critical(this,tr("Error"),tr("Datei existiert nicht"));
        return;
    }

    while (!file.atEnd())
    {
        QString line = file.readLine();
        QStringList strings=line.split('"');

        for(int i=0;i<strings.length();++i)
        {
            if(i%2)
            {
                strings[i].replace(',',"^^^^");
            }
        }
        line=strings.join("");


        QStringList lst=line.split(',');
        if(lst.count()==6)
        {
            bool ok;
            lst[0].toInt(&ok,10);
            if(!ok)
            {
                qDebug()<<lst[0]<<"not valid";
                continue;
            }
            numOfRaws++;
            QString tval=lst[0].simplified();


            QString searchVal=tval;
            searchVal=tval.left(tval.length()-1);



            QTreeWidgetItem* item=new QTreeWidgetItem( 0);


            for(uint i=0;i<6;++i)
            {
                QString val=lst[i].simplified();
                val.replace("^^^^",",");
                item->setText(i, val);
            }
            item->setChildIndicatorPolicy(QTreeWidgetItem::ChildIndicatorPolicy::DontShowIndicatorWhenChildless);

            bool hasParent=false;
            if(searchVal.length()>0)
            {
                QList<QTreeWidgetItem* >found=list->findItems(searchVal,Qt::MatchExactly|Qt::MatchRecursive,0);
                for( int i=0;i<found.count();++i)
                {
                    if(found[i]->text(0)==searchVal)
                    {
//                         qDebug()<<"join "<<lst[0]<<"to"<<found[i]->text(0);
                        hasParent=true;
                        found[i]->addChild(item);
                        break;
                    }
                }
            }
            if(!hasParent)
            {
                list->addTopLevelItem(item);
            }
        }
        else
        {
            qDebug()<<"not valid: "<<lst.count()<<line;
        }
    }

		QIcon fileplanBoardIcon = QIcon::fromTheme("folder-fileplan-board");
		QIcon fileplanShelfIcon = QIcon::fromTheme("folder-fileplan-shelf");

    QTreeWidgetItemIterator it(list);
    while (*it) {
        if ((*it)->childCount())
        {
            (*it)->setIcon(1,fileplanBoardIcon);
        }
        else
        {
            (*it)->setIcon(1,fileplanShelfIcon);
        }
        ++it;
    }

    file.close();
}

void aKtenplan::initTable()
{
    pSwitch->setWhatsThis(tr("Tipp:\n Die Baumansicht ordnet die Aktenzeichen in einer hierarchischen Baumstruktur an.\n Es werden nur Aktenzeichen geoeffnete Gruppierungen angezeigt!"));
    pSwitch->setToolTip(tr("Baumansicht"));
    pSwitch->setText(tr("Baumansich&t"));


    pSwitch->setIcon(QIcon::fromTheme("view-list-tree"));

    table = new ATabView ( numOfRaws,5,this);
    setCentralWidget(table);
    table->show();
    table->setDragEnabled(true);

    QFont* fnt = new QFont();
    fnt->setBold(true);
    QFontMetrics *qm = new QFontMetrics(*fnt);
    int width2=qm->horizontalAdvance(tr(" Beschreibung "));
    fnt->setBold(false);
    QFontMetrics *qm1 = new QFontMetrics(*fnt);

    lastIndex=0;

    int i=0;
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QMessageBox::critical(this,tr("Error"),tr("Datei existiert nicht"));
        return;
    }


    numOfRaws=0;
    while (!file.atEnd())
    {
        QString line = file.readLine();
        QStringList strings=line.split('"');

        for(int i=0;i<strings.length();++i)
        {
            if(i%2)
            {
                strings[i].replace(',',"^^^^");
            }
        }
        line=strings.join("");


        QStringList lst=line.split(',');
        if(lst.count()==6)
        {
            bool ok;
            lst[0].toInt(&ok,10);
            if(!ok)
            {
                qDebug()<<lst[0]<<"not valid";
                continue;
            }
            ++numOfRaws;
            for(uint i=1;i<6;++i)
            {
                lst[i]=lst[i].simplified();
                lst[i].replace("^^^^",",");
                table->setItem(numOfRaws-1,i-1,new QTableWidgetItem(lst[i]));
            }
            table->setVerticalHeaderItem(numOfRaws-1,new QTableWidgetItem(lst[0]));


            int twidth=qm1->horizontalAdvance(lst[1]);
            if(twidth>width2)
                width2=twidth;
            i++;

        }
        else
        {
            qDebug()<<"not valid: "<<lst.count()<<line;
        }
    }

    file.close();



//     table->resizeColumnsToContents();
    table->setHorizontalHeaderItem(0,new QTableWidgetItem(tr("Beschreibung")));
    table->horizontalHeaderItem(0)->setTextAlignment(Qt::AlignmentFlag::AlignLeft);
    table->setHorizontalHeaderItem(1,new QTableWidgetItem(tr("ABFK")));
    table->setHorizontalHeaderItem(2,new QTableWidgetItem(tr("ABFL")));
    table->setHorizontalHeaderItem(3,new QTableWidgetItem(tr("Bemerkungen")));
    table->setHorizontalHeaderItem(4,new QTableWidgetItem(tr("Fundstellen")));

}


bool aKtenplan::slotGetFname()
{

    QString fileName=QFileDialog::getOpenFileName(this, tr("Open - aKtenplan"), "/usr/share/aktenplan",
                                                  tr("*.csv|CSV Dateien (*.csv)\n*.*|Alle Dateien (*.*)"));

    if(!fileName.isNull())
    {
        this->fileName=fileName;
        saveFname();
        return true;
    }
    return false;
}

void aKtenplan::saveFname()
{
    mConfig->setValue("fname",fileName);
}


void aKtenplan::slotAbout()
{
    QString title=tr("aKtenplan ein Desktop - Aktenplan");
    QString about="2.0.0<br>(C) 2005-2018 Stadt Treuchtlingen info@treuchtlingen.de";
    QMessageBox::about(this, tr("about aKtenplan"),about);
}


void aKtenplan::slotAboutQt()
{

    QMessageBox::aboutQt(this);
}



void aKtenplan::slotSwitchView()
{
    if(!tableView)
    {
        delete list;
        list=0;
        initTable();
    }
    else
    {
        delete table;
        table=0;
        initList();
    }
    tableView= !tableView;
}


/*!
 *    \fn aKtenplan::slotCopy()
 */
void aKtenplan::slotCopy()
{
    QClipboard *cb = QApplication::clipboard();
    if(!tableView)
    {
        QTreeWidgetItem* item=list->currentItem();
        if(item)
        {
            cb->setText(item->text(0)+"\t"+item->text(1),QClipboard::Clipboard);
        }
    }
    else
    {
        int csel=table->currentRow();
        if(csel>=0)
        {
            QString txt=table->verticalHeaderItem(csel)->text()+"\t"+
            table->currentItem()->text();
            cb->setText(txt,QClipboard::Clipboard);

        }

    }
}


void aKtenplan::slotCopyAll()
{
    QString txt;
    QClipboard *cb = QApplication::clipboard();
    if(!tableView)
    {
        QList<QTreeWidgetItem*> items=list->selectedItems();
        for(int i=0;i<items.count();++i)
        {
            txt+=items[i]->text(0)+"\t"+
            items[i]->text(1)+"\n";
        }
    }
    else
    {
        QList<QTableWidgetItem*> items=table->selectedItems();
        for(int i=0;i<items.count();++i)
        {
            txt+=table->verticalHeaderItem(items[i]->row())->text()+"\t"+
            items[i]->text()+"\n";

        }
    }
    if(txt!="")
        cb->setText(txt,QClipboard::Clipboard);
}


void aKtenplan::slotSearchText(const QString& text)
{
    QString msg="0";
    msg+=tr(" Suchergebnisse");
    mStatus->setText(msg);
    if(text != "")
        findButton->setEnabled(true);
    else
        findButton->setEnabled(false);
    findButton->setToolTip(tr("Suchen"));
    findButton->setIcon(QIcon::fromTheme("edit-find"));
    if(tableView)
        table->clearSelection();
    else
    {
        closeList();
        list->clearSelection();
        if(text.length()>3)
        {
            listOpen(text);
        }
    }
    lastIndex=0;
    lastItem=0;
}


/*!
 *    \fn aKtenplan::slotFind()
 */
void aKtenplan::slotFind()
{
    QString searched=lineEdit->text();
    if(tableView)
    {
        for(int i=lastIndex;i<numOfRaws;i++)
        {
            QString str=table->item(i,0)->text();
            if(str.contains(searched,Qt::CaseInsensitive))
            {
                lastIndex=i+1;
                table->clearSelection();
                table->selectRow(i);
                table->scrollToItem(table->item(i,0));
                findButton->setToolTip(tr("Weitersuchen"));
                findButton->setIcon(QIcon::fromTheme("go-next"));
                return;
            }
        }
        lastIndex=0;
        findButton->setToolTip(tr("Suchen"));
        findButton->setIcon(QIcon::fromTheme("edit-find"));
        QMessageBox::information( this,tr("Suche"),
                                  tr("Begriff nicht gefunden.") );
    }
    else
    {
        list->clearSelection();
        QTreeWidgetItemIterator it(list);
        if(lastItem!=0l)
            it=QTreeWidgetItemIterator(lastItem);
        it++;
        while ( (*it))
        {
            if((*it)->text(1).contains(searched,Qt::CaseInsensitive))
            {
                list->scrollToItem((*it));
                (*it)->setSelected(true);
                list->setCurrentItem(*it);
                lastItem=(*it);
                findButton->setToolTip(tr("Weitersuchen"));
                findButton->setIcon(QIcon::fromTheme("go-next"));
                return;
            }
            ++it;
        }
        lastItem=0;
        findButton->setToolTip(tr("Suchen"));
        findButton->setIcon(QIcon::fromTheme("edit-find"));
        QMessageBox::information( this,tr("Suche"),
                                  tr("Begriff nicht gefunden.") );
    }
}


void aKtenplan::slotOpenList()
{
    QTreeWidgetItemIterator it(list);
    int found=0;
    QTreeWidgetItem* first=0;
    while (*it)
    {
        //         qDebug()<<text<<(*it)->text(1);
        if((*it)->text(1).contains(searchText,Qt::CaseSensitivity::CaseInsensitive))
        {
            if(!first)
                first=(*it);
            list->scrollToItem((*it));
            (*it)->setSelected(true);

            QTreeWidgetItem* parent=(*it)->parent();
            while(parent)
            {
                parent->setExpanded(true);
                parent=parent->parent();
            }

            found++;
        }
        ++it;
    }
    list->scrollToItem(first);

    QString msg;
    msg.setNum(found);
    msg+=tr(" Suchergebnisse");
    mStatus->setText(msg);
}


/*!
 *    \fn aKtenplan::listOpen(const QString& text)
 */
void aKtenplan::listOpen(const QString& text)
{

    searchText=text;
    slotOpenList();

}


/*!
 *    \fn aKtenplan::closeList()
 */
void aKtenplan::closeList()
{
    QTreeWidgetItemIterator it(list);
    while (*it)
    {
        (*it)->setExpanded(false);
        ++it;
    }
}

void aKtenplan::contextMenuEvent(QContextMenuEvent *event)
{
    qDebug()<<"context!!!";
    QMenu contextMenu(this);
    contextMenu.addAction(copy);
    contextMenu.addAction(copyAll);
    contextMenu.addAction(openFolder);
    contextMenu.exec(event->globalPos());
}


void aKtenplan::slotFOpen()
{
    if(!slotGetFname())
        return;
    if(!tableView)
    {
        delete list;
        list=0;
    }
    else
    {
        delete table;
        table=0;
    }
    tableView=false;
    initList();
}

void aKtenplan::createIndexFile(QString path, bool board)
{
    QFile file(path+"/.directory");
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
    {
        qDebug()<<"can't open for writing"<<path+"/.directory";
        return;
    }

    QTextStream out(&file);
    out << "[Desktop Entry]\n";
    if(!board)
        out<<"Icon=folder-fileplan-shelf\n";
    else
        out<<"Icon=folder-fileplan-board\n";
    file.close();

}


void aKtenplan::initDir(QTreeWidgetItem* it, QString base, QString path)
{
    QString dirname=it->text(0);
    QDir dir(path);
    if(!dir.mkdir(dirname))
    {
        qDebug()<<"Can't create dir"<<dirname;
    }

    QString fname=base+"/"+dirname+" - "+it->text(1);

    fname.replace("ä","ae");
    fname.replace("ü","ue");
    fname.replace("ö","oe");
    fname.replace("Ä","Ae");
    fname.replace("Ü","Ue");
    fname.replace("Ö","Oe");
    fname.replace("ß","ss");

    if(fname.length()>256)
        fname=fname.left(255);

    if(!QFile::link(path+"/"+dirname, fname))
    {
        qDebug()<<"Can't create link"<<path+"/"+dirname<< fname;

    }
    createIndexFile(path+"/"+dirname, it->childCount());

    for(int i=0;i<it->childCount();++i)
    {
        initDir(it->child(i),base,path+"/"+dirname);
    }
}


void aKtenplan::slotCreateFolder()
{
    if(tableView)
        slotSwitchView();
    QString dir = QFileDialog::getExistingDirectory(this, tr("Open Directory where you want to create plan structure"),
                                                    rootURL,
                                                    QFileDialog::ShowDirsOnly
                                                    | QFileDialog::DontResolveSymlinks);



    if(dir.length()>0)
    {
        rootURL=dir;
        mConfig->setValue("root",rootURL);

        QDir dr(dir);

        createIndexFile(dir,true);
        dr.mkdir(".aktenplan");
        createIndexFile(dir+"/.aktenplan",true);
        for(int i=0;i<list->topLevelItemCount();++i)
        {
            initDir(list->topLevelItem(i), dir, dir+"/.aktenplan");
        }
    }

}



void aKtenplan::slotOpenFolder()
{
    qDebug()<<"open folder"<<rootURL;
    if(rootURL.length()<=0)
    {
        QMessageBox::information( this,tr("aKtenplan"),
                                  tr("Please select file plan root first"));
        slotSelectRoot();
    }

    if(!tableView)
        searchVal= list->currentItem()->text(0);
    else
        searchVal=table->verticalHeaderItem(table->currentRow())->text();
    currentVal="";

    qDebug()<<"open folder"<<searchVal;

    dir("/");
}



void aKtenplan::slotSelectRoot()
{
    QString dir = QFileDialog::getExistingDirectory(this, tr("Open Directory"),
                                                    rootURL,
                                                    QFileDialog::ShowDirsOnly
                                                    | QFileDialog::DontResolveSymlinks);
    if(dir.length()>0)
    {
        rootURL=dir;
        mConfig->setValue("root",rootURL);
    }
}

void aKtenplan::dir(QString path)
{
    currentPath=path;
    QDir dr(rootURL+path);
    QStringList dirs=dr.entryList(QDir::Dirs);

    qDebug()<<"search for"<<searchVal<<"in"<<rootURL+path;
    QString dname=searchVal+" - ";
    for(int i=0;i<dirs.count();++i)
    {
        if(dirs[i].indexOf(dname)==0)
        {
            qDebug()<<"found dir: "<<dirs[i];
               launchKonqueror(currentPath+"/"+dirs[i]);
               return;
        }
    }
    launchKonqueror(currentPath);
}



void aKtenplan::launchKonqueror(QString path)
{
    QStringList args;
    args<<rootURL+path;
    if(!QProcess::startDetached("dolphin",args))
    {
        QMessageBox::critical(this,"Fehler", "Starten des Dateimanagers fehlgeschlagen");
    }
}

