import 'package:flutter/material.dart';
import 'package:aspiro/models/application.dart';
import 'package:aspiro/services/config_service.dart';
import 'package:aspiro/services/xml_service.dart';
import 'package:aspiro/widgets/side_bar.dart';
import 'package:aspiro/widgets/xml_viewer.dart';

class HomeScreen extends StatefulWidget {
  const HomeScreen({super.key});

  @override
  State<HomeScreen> createState() => _HomeScreenState();
}

class _HomeScreenState extends State<HomeScreen> {
  List<ApplicationGroup> _applicationGroups = [];
  Application? _selectedApplication;
  bool _isLoading = true;
  String? _error;

  @override
  void initState() {
    super.initState();
    _initializeApp();
  }

  Future<void> _initializeApp() async {
    try {
      await ConfigService.instance.loadConfig();
      await _loadApplications();
    } catch (e) {
      setState(() {
        _error = 'Fehler beim Laden der Anwendung: $e';
        _isLoading = false;
      });
    }
  }

  Future<void> _loadApplications() async {
    try {
      setState(() => _isLoading = true);
      final List<ApplicationGroup> groups = await XmlService.instance
          .loadApplications();
      setState(() {
        _applicationGroups = groups;
        _isLoading = false;
        _error = null;

        // Update selected application if it exists in the new data
        if (_selectedApplication != null) {
          _selectedApplication = _findUpdatedApplication(_selectedApplication!);
        }
      });
    } catch (e) {
      setState(() {
        _error = 'Fehler beim Laden der XML-Dateien: $e';
        _isLoading = false;
      });
    }
  }

  Application? _findUpdatedApplication(Application oldApplication) {
    for (final ApplicationGroup group in _applicationGroups) {
      for (final Application app in group.applications) {
        if (app.fileName == oldApplication.fileName) {
          return app;
        }
      }
    }
    return null;
  }

  void _onApplicationSelected(Application application) {
    setState(() {
      _selectedApplication = application;
    });
  }

  void _onApplicationUpdated(Application updatedApplication) {
    // Update the application in the groups
    for (int i = 0; i < _applicationGroups.length; i++) {
      final ApplicationGroup group = _applicationGroups[i];
      for (int j = 0; j < group.applications.length; j++) {
        if (group.applications[j].fileName == updatedApplication.fileName) {
          _applicationGroups[i].applications[j] = updatedApplication;
          break;
        }
      }
    }

    setState(() {
      _selectedApplication = updatedApplication;
    });
  }

  void _onApplicationDeleted() {
    // Remove the application from groups and reload
    setState(() {
      _selectedApplication = null;
    });
    _loadApplications();
  }

  void _goHome() {
    setState(() {
      _selectedApplication = null;
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.home),
          onPressed: _goHome,
          tooltip: 'Startseite',
        ),
        title: Text(ConfigService.instance.appTitle),
        actions: [
          IconButton(
            icon: const Icon(Icons.refresh),
            onPressed: _loadApplications,
            tooltip: 'Aktualisieren',
          ),
          IconButton(
            icon: const Icon(Icons.settings),
            onPressed: () => _showSettings(context),
            tooltip: 'Einstellungen',
          ),
        ],
      ),
      body: _buildBody(),
    );
  }

  Widget _buildBody() {
    if (_isLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    if (_error != null) {
      return Center(
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(
              Icons.error_outline,
              size: 64,
              color: Theme.of(context).colorScheme.error,
            ),
            const SizedBox(height: 16),
            Text(
              _error!,
              style: TextStyle(
                color: Theme.of(context).colorScheme.error,
                fontSize: 16,
              ),
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 16),
            ElevatedButton(
              onPressed: _loadApplications,
              child: const Text('Erneut versuchen'),
            ),
          ],
        ),
      );
    }

    return Row(
      children: [
        Sidebar(
          groups: _applicationGroups,
          selectedApplication: _selectedApplication,
          onApplicationSelected: _onApplicationSelected,
        ),
        Expanded(
          child: XmlViewer(
            applicationGroups: _applicationGroups,
            application: _selectedApplication,
            onApplicationUpdated: _onApplicationUpdated,
            onApplicationDeleted: _onApplicationDeleted,
            onApplicationSelected: _onApplicationSelected,
          ),
        ),
      ],
    );
  }

  void _showSettings(BuildContext context) {
    final int totalApplications = _applicationGroups.fold<int>(
      0,
      (sum, group) => sum + group.applications.length,
    );

    final Map<ProcessingStatus, int> statusCounts = {};
    for (final ApplicationGroup group in _applicationGroups) {
      for (final Application app in group.applications) {
        final ProcessingStatus status = app.internalProcessing.status;
        statusCounts[status] = (statusCounts[status] ?? 0) + 1;
      }
    }

    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        title: const Text('Einstellungen & Statistiken'),
        content: SizedBox(
          width: 500,
          child: SingleChildScrollView(
            child: Column(
              mainAxisSize: MainAxisSize.min,
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                // Konfigurationsstatus
                Container(
                  padding: const EdgeInsets.all(12),
                  decoration: BoxDecoration(
                    color: ConfigService.instance.isConfigLoaded
                        ? Colors.green[50]
                        : Colors.orange[50],
                    borderRadius: BorderRadius.circular(8),
                    border: Border.all(
                      color: ConfigService.instance.isConfigLoaded
                          ? Colors.green[200]!
                          : Colors.orange[200]!,
                    ),
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Row(
                        children: [
                          Icon(
                            ConfigService.instance.isConfigLoaded
                                ? Icons.check_circle
                                : Icons.warning,
                            color: ConfigService.instance.isConfigLoaded
                                ? Colors.green[700]
                                : Colors.orange[700],
                            size: 20,
                          ),
                          const SizedBox(width: 8),
                          Text(
                            'Konfigurationsstatus',
                            style: TextStyle(
                              fontWeight: FontWeight.bold,
                              color: ConfigService.instance.isConfigLoaded
                                  ? Colors.green[700]
                                  : Colors.orange[700],
                            ),
                          ),
                        ],
                      ),
                      const SizedBox(height: 8),
                      Text(
                        ConfigService.instance.isConfigLoaded
                            ? '✅ assets/config.json erfolgreich geladen'
                            : '⚠️ Fallback-Werte verwendet',
                        style: const TextStyle(fontSize: 12),
                      ),
                      if (!ConfigService.instance.isConfigLoaded &&
                          ConfigService.instance.loadError != null) ...[
                        const SizedBox(height: 4),
                        Text(
                          'Fehler: ${ConfigService.instance.loadError}',
                          style: TextStyle(
                            fontSize: 11,
                            color: Colors.red[600],
                            fontFamily: 'monospace',
                          ),
                        ),
                      ],
                    ],
                  ),
                ),
                const SizedBox(height: 16),

                // Aktuelle Konfiguration
                const Text(
                  'Aktuelle Konfiguration:',
                  style: TextStyle(fontWeight: FontWeight.bold),
                ),
                const SizedBox(height: 8),
                _buildSettingsRow(
                  'XML-Verzeichnis:',
                  ConfigService.instance.xmlDirectoryPath,
                ),
                _buildSettingsRow(
                  'App-Titel:',
                  ConfigService.instance.appTitle,
                ),
                _buildSettingsRow(
                  'Theme-Modus:',
                  ConfigService.instance.themeMode,
                ),
                _buildSettingsRow(
                  'Löschfrist:',
                  '${ConfigService.instance.deleteAfterMonths} Monate',
                ),
                const SizedBox(height: 8),
                _buildSettingsRow(
                  'Geladene Bewerbungen:',
                  totalApplications.toString(),
                ),

                const SizedBox(height: 16),

                // Raw Config (für Debug)
                ExpansionTile(
                  title: const Text(
                    'Raw Config (Debug)',
                    style: TextStyle(fontSize: 14, fontWeight: FontWeight.w500),
                  ),
                  children: [
                    Container(
                      width: double.infinity,
                      padding: const EdgeInsets.all(12),
                      decoration: BoxDecoration(
                        color: Colors.grey[100],
                        borderRadius: BorderRadius.circular(4),
                      ),
                      child: Text(
                        ConfigService.instance.rawConfig?.toString() ??
                            'Keine Konfiguration geladen',
                        style: const TextStyle(
                          fontFamily: 'monospace',
                          fontSize: 11,
                        ),
                      ),
                    ),
                  ],
                ),

                const SizedBox(height: 16),
                const Text(
                  'Status-Übersicht:',
                  style: TextStyle(fontWeight: FontWeight.bold),
                ),
                const SizedBox(height: 8),
                ...statusCounts.entries.map(
                  (entry) => Padding(
                    padding: const EdgeInsets.symmetric(vertical: 2),
                    child: Row(
                      children: [
                        Container(
                          width: 12,
                          height: 12,
                          decoration: BoxDecoration(
                            color: _getStatusColor(entry.key),
                            shape: BoxShape.circle,
                          ),
                        ),
                        const SizedBox(width: 8),
                        Expanded(child: Text(_getStatusText(entry.key))),
                        Text('${entry.value}'),
                      ],
                    ),
                  ),
                ),

                const SizedBox(height: 16),

                // Anleitung
                Container(
                  padding: const EdgeInsets.all(12),
                  decoration: BoxDecoration(
                    color: Colors.blue[50],
                    borderRadius: BorderRadius.circular(8),
                    border: Border.all(color: Colors.blue[200]!),
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Row(
                        children: [
                          Icon(Icons.info, color: Colors.blue[700], size: 20),
                          const SizedBox(width: 8),
                          Text(
                            'Konfiguration anpassen',
                            style: TextStyle(
                              fontWeight: FontWeight.bold,
                              color: Colors.blue[700],
                            ),
                          ),
                        ],
                      ),
                      const SizedBox(height: 8),
                      const Text(
                        'Um die Konfiguration zu ändern:\n'
                        '1. Bearbeite assets/config.json in deinem Projekt\n'
                        '2. Führe "flutter clean" aus\n'
                        '3. Starte die App neu mit "flutter run"\n'
                        '4. Die neuen Werte werden automatisch geladen',
                        style: TextStyle(fontSize: 12),
                      ),
                    ],
                  ),
                ),
              ],
            ),
          ),
        ),
        actions: [
          TextButton(
            onPressed: () => Navigator.of(context).pop(),
            child: const Text('Schließen'),
          ),
        ],
      ),
    );
  }

  Widget _buildSettingsRow(String label, String value) {
    return Row(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        SizedBox(
          width: 140,
          child: Text(
            label,
            style: const TextStyle(fontWeight: FontWeight.w600),
          ),
        ),
        Expanded(
          child: Text(value, style: const TextStyle(fontFamily: 'monospace')),
        ),
      ],
    );
  }

  Color _getStatusColor(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return Colors.red[400]!;
      case ProcessingStatus.unbearbeitet:
        return Colors.grey[400]!;
      case ProcessingStatus.geprueft:
        return Colors.blue[400]!;
      case ProcessingStatus.eingeladen:
        return Colors.orange[400]!;
      case ProcessingStatus.gespraechGefuehrt:
        return Colors.purple[400]!;
      case ProcessingStatus.zusage:
        return Colors.green[400]!;
      case ProcessingStatus.absage:
        return Colors.red[300]!;
    }
  }

  String _getStatusText(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return 'Ungültig';
      case ProcessingStatus.unbearbeitet:
        return 'Neu';
      case ProcessingStatus.geprueft:
        return 'Geprüft';
      case ProcessingStatus.eingeladen:
        return 'Eingeladen';
      case ProcessingStatus.gespraechGefuehrt:
        return 'Gespräch geführt';
      case ProcessingStatus.zusage:
        return 'Zusage';
      case ProcessingStatus.absage:
        return 'Absage';
    }
  }
}
