import 'package:flutter/material.dart';
import 'package:aspiro/models/application.dart';

class Sidebar extends StatelessWidget {
  final List<ApplicationGroup> groups;
  final Application? selectedApplication;
  final Function(Application) onApplicationSelected;

  const Sidebar({
    super.key,
    required this.groups,
    required this.selectedApplication,
    required this.onApplicationSelected,
  });

  @override
  Widget build(BuildContext context) {
    return Container(
      width: 380,
      decoration: BoxDecoration(
        color: Theme.of(context).colorScheme.surface,
        border: Border(
          right: BorderSide(
            color: Theme.of(context).colorScheme.outline.withOpacity(0.2),
          ),
        ),
      ),
      child: Column(
        children: [
          Container(
            padding: const EdgeInsets.all(16),
            decoration: BoxDecoration(
              color: Theme.of(context).colorScheme.surfaceContainerHighest,
              border: Border(
                bottom: BorderSide(
                  color: Theme.of(context).colorScheme.outline.withOpacity(0.2),
                ),
              ),
            ),
            child: Row(
              children: [
                Icon(
                  Icons.folder_open,
                  color: Theme.of(context).colorScheme.primary,
                ),
                const SizedBox(width: 8),
                Text(
                  'Bewerbungen',
                  style: Theme.of(
                    context,
                  ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
                ),
              ],
            ),
          ),
          Expanded(
            child: groups.isEmpty
                ? const Center(child: Text('Keine Bewerbungen gefunden'))
                : ListView.builder(
                    itemCount: groups.length,
                    itemBuilder: (context, index) =>
                        _buildGroupTile(context, groups[index]),
                  ),
          ),
        ],
      ),
    );
  }

  Widget _buildGroupTile(BuildContext context, ApplicationGroup group) {
    if (group.applications.length == 1) {
      return _buildApplicationTile(context, group.applications.first);
    }

    return ExpansionTile(
      leading: Icon(Icons.work, color: Theme.of(context).colorScheme.primary),
      title: Text(
        group.stellentitel,
        style: const TextStyle(fontWeight: FontWeight.w600),
      ),
      subtitle: Text('${group.applications.length} Bewerbungen'),
      children: group.applications
          .map(
            (app) => Padding(
              padding: const EdgeInsets.only(left: 16),
              child: _buildApplicationTile(context, app),
            ),
          )
          .toList(),
    );
  }

  Widget _buildApplicationTile(BuildContext context, Application application) {
    final bool isSelected = selectedApplication == application;
    final ProcessingStatus status = application.internalProcessing.status;
    final bool hasGdB = application.internalProcessing.gdb > 0;
    final bool hasZusage = application.internalProcessing.zusage;

    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 8, vertical: 2),
      decoration: BoxDecoration(
        color: isSelected
            ? Theme.of(context).colorScheme.primaryContainer
            : null,
        borderRadius: BorderRadius.circular(8),
      ),
      child: ListTile(
        leading: Stack(
          children: [
            CircleAvatar(
              backgroundColor: isSelected
                  ? Theme.of(context).colorScheme.primary
                  : Theme.of(context).colorScheme.secondary,
              child: Text(
                application.vorname.isNotEmpty
                    ? application.vorname[0].toUpperCase()
                    : '?',
                style: TextStyle(
                  color: isSelected
                      ? Theme.of(context).colorScheme.onPrimary
                      : Theme.of(context).colorScheme.onSecondary,
                  fontWeight: FontWeight.bold,
                ),
              ),
            ),
            Positioned(
              right: -2,
              bottom: -2,
              child: Container(
                width: 16,
                height: 16,
                decoration: BoxDecoration(
                  color: _getStatusColor(status),
                  shape: BoxShape.circle,
                  border: Border.all(
                    color: Theme.of(context).colorScheme.surface,
                    width: 2,
                  ),
                ),
                child: _getStatusIcon(status),
              ),
            ),
          ],
        ),
        title: Text(
          application.displayName,
          style: TextStyle(
            fontWeight: FontWeight.w600,
            color: isSelected
                ? Theme.of(context).colorScheme.onPrimaryContainer
                : null,
          ),
        ),
        subtitle: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              application.displayTitle,
              style: TextStyle(
                color: isSelected
                    ? Theme.of(
                        context,
                      ).colorScheme.onPrimaryContainer.withOpacity(0.7)
                    : null,
              ),
            ),
            const SizedBox(height: 4),
            Wrap(
              spacing: 4,
              runSpacing: 2,
              children: [
                Container(
                  padding: const EdgeInsets.symmetric(
                    horizontal: 6,
                    vertical: 2,
                  ),
                  decoration: BoxDecoration(
                    color: _getStatusColor(status),
                    borderRadius: BorderRadius.circular(10),
                  ),
                  child: Text(
                    _getStatusText(status),
                    style: TextStyle(
                      color: _getStatusTextColor(status),
                      fontSize: 10,
                      fontWeight: FontWeight.w600,
                    ),
                  ),
                ),
                if (hasZusage)
                  Container(
                    padding: const EdgeInsets.symmetric(
                      horizontal: 6,
                      vertical: 2,
                    ),
                    decoration: BoxDecoration(
                      color: Colors.green[600],
                      borderRadius: BorderRadius.circular(10),
                    ),
                    child: Row(
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        const Icon(
                          Icons.check_circle,
                          size: 10,
                          color: Colors.white,
                        ),
                        const SizedBox(width: 2),
                        const Text(
                          'ZUSAGE',
                          style: TextStyle(
                            color: Colors.white,
                            fontSize: 9,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ],
                    ),
                  ),
                if (hasGdB)
                  Container(
                    padding: const EdgeInsets.symmetric(
                      horizontal: 6,
                      vertical: 2,
                    ),
                    decoration: BoxDecoration(
                      color: Colors.purple[600],
                      borderRadius: BorderRadius.circular(10),
                    ),
                    child: Row(
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        const Icon(
                          Icons.accessible,
                          size: 10,
                          color: Colors.white,
                        ),
                        const SizedBox(width: 2),
                        Text(
                          'GdB ${application.internalProcessing.gdb}%',
                          style: const TextStyle(
                            color: Colors.white,
                            fontSize: 9,
                            fontWeight: FontWeight.w600,
                          ),
                        ),
                      ],
                    ),
                  ),
                if (application.internalProcessing.gueltig &&
                    application.internalProcessing.istEingeladen)
                  Container(
                    padding: const EdgeInsets.symmetric(
                      horizontal: 4,
                      vertical: 2,
                    ),
                    decoration: BoxDecoration(
                      color: Colors.orange[100],
                      borderRadius: BorderRadius.circular(8),
                    ),
                    child: Icon(
                      Icons.event,
                      size: 10,
                      color: Colors.orange[700],
                    ),
                  ),
              ],
            ),
          ],
        ),
        onTap: () => onApplicationSelected(application),
      ),
    );
  }

  Color _getStatusColor(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return Colors.red[400]!;
      case ProcessingStatus.unbearbeitet:
        return Colors.grey[400]!;
      case ProcessingStatus.geprueft:
        return Colors.blue[400]!;
      case ProcessingStatus.eingeladen:
        return Colors.orange[400]!;
      case ProcessingStatus.gespraechGefuehrt:
        return Colors.purple[400]!;
      case ProcessingStatus.zusage:
        return Colors.green[400]!;
      case ProcessingStatus.absage:
        return Colors.red[300]!;
    }
  }

  Color _getStatusTextColor(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
      case ProcessingStatus.zusage:
      case ProcessingStatus.absage:
        return Colors.white;
      default:
        return Colors.white;
    }
  }

  Widget _getStatusIcon(ProcessingStatus status) {
    IconData iconData;
    switch (status) {
      case ProcessingStatus.ungueltig:
        iconData = Icons.close;
        break;
      case ProcessingStatus.unbearbeitet:
        iconData = Icons.schedule;
        break;
      case ProcessingStatus.geprueft:
        iconData = Icons.visibility;
        break;
      case ProcessingStatus.eingeladen:
        iconData = Icons.event;
        break;
      case ProcessingStatus.gespraechGefuehrt:
        iconData = Icons.chat;
        break;
      case ProcessingStatus.zusage:
        iconData = Icons.check;
        break;
      case ProcessingStatus.absage:
        iconData = Icons.close;
        break;
    }

    return Icon(iconData, size: 10, color: _getStatusTextColor(status));
  }

  String _getStatusText(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return 'Ungültig';
      case ProcessingStatus.unbearbeitet:
        return 'Neu';
      case ProcessingStatus.geprueft:
        return 'Geprüft';
      case ProcessingStatus.eingeladen:
        return 'Eingeladen';
      case ProcessingStatus.gespraechGefuehrt:
        return 'Gespräch';
      case ProcessingStatus.zusage:
        return 'Zusage';
      case ProcessingStatus.absage:
        return 'Absage';
    }
  }
}
