
class DocumentationEntry {
  final String typ;
  final String beschreibung;
  final String anwender;
  final DateTime zeitstempel;

  DocumentationEntry({
    required this.typ,
    required this.beschreibung,
    required this.anwender,
    required this.zeitstempel,
  });

  DocumentationEntry.fromXml(
    String typ,
    String beschreibung,
    String anwender,
    String zeitstempel,
  ) : typ = typ,
      beschreibung = beschreibung,
      anwender = anwender,
      zeitstempel = DateTime.tryParse(zeitstempel) ?? DateTime.now();
}

class InternalProcessing {
  bool gueltig;
  String geprueftAnwender;
  DateTime? geprueftDatum;
  DateTime? eingeladenDatum;
  String eingeladenUhrzeit;
  bool wahrgenommen;
  int ausbildungBewertung; // 1-5
  int motivationBewertung; // 1-5
  int gehaltsvorstellungenBewertung; // 1-5
  int teamakzeptanzBewertung; // 1-5
  int gdb; // Grad der Behinderung 0-100%
  bool zusage;
  List<DocumentationEntry> dokumentation;

  InternalProcessing({
    this.gueltig = false,
    this.geprueftAnwender = '',
    this.geprueftDatum,
    this.eingeladenDatum,
    this.eingeladenUhrzeit = '',
    this.wahrgenommen = false,
    this.ausbildungBewertung = 0,
    this.motivationBewertung = 0,
    this.gehaltsvorstellungenBewertung = 0,
    this.teamakzeptanzBewertung = 0,
    this.gdb = 0,
    this.zusage = false,
    this.dokumentation = const [],
  });

  bool get istGeprueft => geprueftAnwender.isNotEmpty && geprueftDatum != null;
  bool get istEingeladen =>
      eingeladenDatum != null && eingeladenUhrzeit.isNotEmpty;
  bool get istBewertet =>
      ausbildungBewertung > 0 &&
      motivationBewertung > 0 &&
      gehaltsvorstellungenBewertung > 0 &&
      teamakzeptanzBewertung > 0;

  ProcessingStatus get status {
    if (!gueltig) return ProcessingStatus.ungueltig;
    if (!istGeprueft) return ProcessingStatus.unbearbeitet;
    if (!istEingeladen) return ProcessingStatus.geprueft;
    if (!wahrgenommen) return ProcessingStatus.eingeladen;
    if (!istBewertet) return ProcessingStatus.gespraechGefuehrt;
    if (zusage) return ProcessingStatus.zusage;
    return ProcessingStatus.absage;
  }
}

enum ProcessingStatus {
  ungueltig,
  unbearbeitet,
  geprueft,
  eingeladen,
  gespraechGefuehrt,
  zusage,
  absage,
}

class Application {
  final String fileName;
  final String filePath;
  final String stellentitel;
  final String vorname;
  final String familienname;
  final String anrede;
  final String telefon1;
  final String email;
  final String motivation;
  final String faehigkeiten;
  final Map<String, String> allFields;
  final InternalProcessing internalProcessing;
  final DateTime? timestamp;

  Application({
    required this.fileName,
    required this.filePath,
    required this.stellentitel,
    required this.vorname,
    required this.familienname,
    required this.anrede,
    required this.telefon1,
    required this.email,
    required this.motivation,
    required this.faehigkeiten,
    required this.allFields,
    required this.internalProcessing,
    this.timestamp,
  });

  String get displayName => '$vorname $familienname';

  String get displayTitle => anrede == 'herr'
      ? 'Herr'
      : anrede == 'frau'
      ? 'Frau'
      : anrede.isNotEmpty
      ? anrede
      : '';

  bool canBeDeleted(int deleteAfterMonths) {
    if (timestamp == null) return false;
    final DateTime deleteAfter = timestamp!.add(
      Duration(days: deleteAfterMonths * 30),
    );
    return DateTime.now().isAfter(deleteAfter);
  }

  Application copyWith({InternalProcessing? internalProcessing}) {
    return Application(
      fileName: fileName,
      filePath: filePath,
      stellentitel: stellentitel,
      vorname: vorname,
      familienname: familienname,
      anrede: anrede,
      telefon1: telefon1,
      email: email,
      motivation: motivation,
      faehigkeiten: faehigkeiten,
      allFields: allFields,
      internalProcessing: internalProcessing ?? this.internalProcessing,
      timestamp: timestamp,
    );
  }
}

class ApplicationGroup {
  final String stellentitel;
  final List<Application> applications;

  ApplicationGroup({required this.stellentitel, required this.applications});
}
