import 'dart:convert';
import 'package:flutter/services.dart';

class ConfigService {
  static ConfigService? _instance;
  static ConfigService get instance => _instance ??= ConfigService._();
  ConfigService._();

  Map<String, dynamic>? _config;
  bool _configLoaded = false;
  String? _loadError;

  Future<void> loadConfig() async {
    try {
      print(' Lade Konfiguration aus assets/config.json...');
      final String configString = await rootBundle.loadString(
        'assets/config.json',
      );
      _config = json.decode(configString);
      _configLoaded = true;
      _loadError = null;
      print('✅ Konfiguration erfolgreich geladen: $_config');
    } catch (e) {
      _loadError = e.toString();
      print('❌ Fehler beim Laden der Konfiguration: $e');
      print(' Verwende Standard-Fallback-Werte');
      _config = {
        'xml_directory_path': '/usr/local/fileplan/bewerbungen',
        'app_title': 'Bewerbungs-Viewer',
        'theme_mode': 'light',
        'delete_after_months': 6,
      };
      _configLoaded = false;
    }
  }

  String get xmlDirectoryPath {
    final String value =
        _config?['xml_directory_path'] ?? '/usr/local/fileplan/bewerbungen';
    print(
      ' XML-Verzeichnis: $value (aus ${_configLoaded ? "config.json" : "Fallback"})',
    );
    return value;
  }

  String get appTitle {
    final String value = _config?['app_title'] ?? 'Bewerbungs-Viewer';
    print(
      ' App-Titel: $value (aus ${_configLoaded ? "config.json" : "Fallback"})',
    );
    return value;
  }

  String get themeMode {
    final String value = _config?['theme_mode'] ?? 'light';
    print(
      ' Theme-Modus: $value (aus ${_configLoaded ? "config.json" : "Fallback"})',
    );
    return value;
  }

  int get deleteAfterMonths {
    final int value = _config?['delete_after_months'] ?? 6;
    print(
      '️ Löschfrist: $value Monate (aus ${_configLoaded ? "config.json" : "Fallback"})',
    );
    return value;
  }

  // Debug-Informationen
  bool get isConfigLoaded => _configLoaded;
  String? get loadError => _loadError;
  Map<String, dynamic>? get rawConfig => _config;

  // Methode zum Speichern der Konfiguration (für zukünftige Erweiterungen)
  Future<bool> saveConfig(Map<String, dynamic> newConfig) async {
    try {
      // Hinweis: assets/ ist read-only, hier würde man normalerweise
      // in das app-spezifische Verzeichnis schreiben
      print(' Konfiguration speichern: $newConfig');
      print('⚠️ Hinweis: assets/config.json ist read-only');
      print(
        ' Für persistente Änderungen sollte eine separate config-Datei verwendet werden',
      );
      return false;
    } catch (e) {
      print('❌ Fehler beim Speichern: $e');
      return false;
    }
  }
}
