import 'dart:io';
import 'package:pdf/pdf.dart';
import 'package:pdf/widgets.dart' as pw;
import 'package:aspiro/models/application.dart';
import 'package:aspiro/services/config_service.dart';

class PdfTemplate {
  static Future<bool> exportBewerbung(Application application) async {
    try {
      final pdf = generateBewerbungsSheet(application);

      // Dateiname: bewerbung_vorname_nachname_datum.pdf
      final String fileName =
          'bewerbung_${application.vorname.toLowerCase()}_${application.familienname.toLowerCase()}_${DateTime.now().toString().split(' ')[0]}.pdf'
              .replaceAll(' ', '_')
              .replaceAll('ä', 'ae')
              .replaceAll('ö', 'oe')
              .replaceAll('ü', 'ue')
              .replaceAll('ß', 'ss');

      // Speichere ins XML-Verzeichnis
      final String xmlDir = ConfigService.instance.xmlDirectoryPath;
      final String filePath = '$xmlDir/$fileName';

      final File file = File(filePath);
      await file.writeAsBytes(await pdf.save());

      print(' PDF erfolgreich gespeichert: $filePath');
      return true;
    } catch (e) {
      print('❌ PDF-Export Fehler: $e');
      return false;
    }
  }

  static pw.Document generateBewerbungsSheet(Application application) {
    final pdf = pw.Document();

    pdf.addPage(
      pw.Page(
        pageFormat: PdfPageFormat.a4,
        margin: const pw.EdgeInsets.all(32),
        build: (context) => pw.Column(
          crossAxisAlignment: pw.CrossAxisAlignment.start,
          children: [
            // Header
            _buildHeader(application),
            pw.SizedBox(height: 24),

            // Bewerberdaten
            _buildBewerberSection(application),
            pw.SizedBox(height: 20),

            // Interne Bearbeitung
            _buildInterneBearbeitungSection(application),
            pw.SizedBox(height: 20),

            // Dokumentation
            if (application.internalProcessing.dokumentation.isNotEmpty)
              _buildDokumentationSection(application),

            pw.Spacer(),

            // Footer
            _buildFooter(),
          ],
        ),
      ),
    );

    return pdf;
  }

  static pw.Widget _buildHeader(Application application) {
    return pw.Container(
      padding: const pw.EdgeInsets.all(16),
      decoration: pw.BoxDecoration(
        color: PdfColors.blue50,
        border: pw.Border.all(color: PdfColors.blue200),
        borderRadius: pw.BorderRadius.circular(8),
      ),
      child: pw.Row(
        mainAxisAlignment: pw.MainAxisAlignment.spaceBetween,
        children: [
          pw.Column(
            crossAxisAlignment: pw.CrossAxisAlignment.start,
            children: [
              pw.Text(
                'BEWERBUNGSÜBERSICHT',
                style: pw.TextStyle(
                  fontSize: 18,
                  fontWeight: pw.FontWeight.bold,
                  color: PdfColors.blue800,
                ),
              ),
              pw.SizedBox(height: 4),
              pw.Text(
                application.displayName,
                style: pw.TextStyle(
                  fontSize: 14,
                  fontWeight: pw.FontWeight.bold,
                ),
              ),
              pw.Text(
                'Bewerbung für: ${application.stellentitel}',
                style: const pw.TextStyle(fontSize: 12),
              ),
            ],
          ),
          pw.Column(
            crossAxisAlignment: pw.CrossAxisAlignment.end,
            children: [
              pw.Text(
                'Status: ${_getStatusText(application.internalProcessing.status)}',
                style: pw.TextStyle(
                  fontSize: 12,
                  fontWeight: pw.FontWeight.bold,
                ),
              ),
              pw.Text(
                'Export: ${DateTime.now().toString().split(' ')[0]}',
                style: const pw.TextStyle(
                  fontSize: 10,
                  color: PdfColors.grey600,
                ),
              ),
            ],
          ),
        ],
      ),
    );
  }

  static pw.Widget _buildBewerberSection(Application application) {
    return pw.Column(
      crossAxisAlignment: pw.CrossAxisAlignment.start,
      children: [
        pw.Text(
          'BEWERBERDATEN',
          style: pw.TextStyle(
            fontSize: 14,
            fontWeight: pw.FontWeight.bold,
            color: PdfColors.blue800,
          ),
        ),
        pw.SizedBox(height: 8),
        pw.Container(
          padding: const pw.EdgeInsets.all(12),
          decoration: pw.BoxDecoration(
            border: pw.Border.all(color: PdfColors.grey300),
            borderRadius: pw.BorderRadius.circular(4),
          ),
          child: pw.Column(
            children: [
              _buildDataRow('Name', application.displayName),
              _buildDataRow('Anrede', application.displayTitle),
              _buildDataRow(
                'Telefon',
                application.telefon1.isEmpty ? '-' : application.telefon1,
              ),
              _buildDataRow(
                'E-Mail',
                application.email.isEmpty ? '-' : application.email,
              ),
              _buildDataRow('Stellentitel', application.stellentitel),
              if (application.motivation.isNotEmpty)
                _buildDataRow('Motivation', application.motivation),
              if (application.faehigkeiten.isNotEmpty)
                _buildDataRow('Fähigkeiten', application.faehigkeiten),
            ],
          ),
        ),
      ],
    );
  }

  static pw.Widget _buildInterneBearbeitungSection(Application application) {
    final processing = application.internalProcessing;

    return pw.Column(
      crossAxisAlignment: pw.CrossAxisAlignment.start,
      children: [
        pw.Text(
          'INTERNE BEARBEITUNG',
          style: pw.TextStyle(
            fontSize: 14,
            fontWeight: pw.FontWeight.bold,
            color: PdfColors.blue800,
          ),
        ),
        pw.SizedBox(height: 8),
        pw.Container(
          padding: const pw.EdgeInsets.all(12),
          decoration: pw.BoxDecoration(
            border: pw.Border.all(color: PdfColors.grey300),
            borderRadius: pw.BorderRadius.circular(4),
          ),
          child: pw.Column(
            children: [
              _buildDataRow('Gültig', processing.gueltig ? 'Ja' : 'Nein'),
              _buildDataRow(
                'Geprüft von',
                processing.geprueftAnwender.isEmpty
                    ? '-'
                    : processing.geprueftAnwender,
              ),
              _buildDataRow(
                'Geprüft am',
                processing.geprueftDatum?.toString().split(' ')[0] ?? '-',
              ),
              _buildDataRow(
                'Einladungsdatum',
                processing.eingeladenDatum?.toString().split(' ')[0] ?? '-',
              ),
              _buildDataRow(
                'Uhrzeit',
                processing.eingeladenUhrzeit.isEmpty
                    ? '-'
                    : processing.eingeladenUhrzeit,
              ),
              _buildDataRow(
                'Wahrgenommen',
                processing.wahrgenommen ? 'Ja' : 'Nein',
              ),
              pw.SizedBox(height: 8),
              pw.Text(
                'BEWERTUNGEN',
                style: pw.TextStyle(fontWeight: pw.FontWeight.bold),
              ),
              pw.SizedBox(height: 4),
              _buildDataRow(
                'Ausbildung',
                _formatRating(processing.ausbildungBewertung),
              ),
              _buildDataRow(
                'Motivation',
                _formatRating(processing.motivationBewertung),
              ),
              _buildDataRow(
                'Gehaltsvorstellungen',
                _formatRating(processing.gehaltsvorstellungenBewertung),
              ),
              _buildDataRow(
                'Teamakzeptanz',
                _formatRating(processing.teamakzeptanzBewertung),
              ),
              if (processing.gdb > 0)
                _buildDataRow('GdB', '${processing.gdb}%'),
              pw.SizedBox(height: 8),
              _buildDataRow(
                'Zusage',
                processing.zusage ? 'JA ✓' : 'Nein',
                valueStyle: processing.zusage
                    ? pw.TextStyle(
                        fontWeight: pw.FontWeight.bold,
                        color: PdfColors.green700,
                      )
                    : null,
              ),
            ],
          ),
        ),
      ],
    );
  }

  static pw.Widget _buildDokumentationSection(Application application) {
    return pw.Column(
      crossAxisAlignment: pw.CrossAxisAlignment.start,
      children: [
        pw.Text(
          'DOKUMENTATION',
          style: pw.TextStyle(
            fontSize: 14,
            fontWeight: pw.FontWeight.bold,
            color: PdfColors.blue800,
          ),
        ),
        pw.SizedBox(height: 8),
        ...application.internalProcessing.dokumentation
            .take(5)
            .map(
              (entry) => pw.Container(
                margin: const pw.EdgeInsets.only(bottom: 4),
                padding: const pw.EdgeInsets.all(8),
                decoration: pw.BoxDecoration(
                  color: PdfColors.grey100,
                  borderRadius: pw.BorderRadius.circular(4),
                ),
                child: pw.Row(
                  crossAxisAlignment: pw.CrossAxisAlignment.start,
                  children: [
                    pw.Container(
                      width: 60,
                      child: pw.Text(
                        entry.typ,
                        style: pw.TextStyle(
                          fontSize: 10,
                          fontWeight: pw.FontWeight.bold,
                        ),
                      ),
                    ),
                    pw.SizedBox(width: 8),
                    pw.Expanded(
                      child: pw.Column(
                        crossAxisAlignment: pw.CrossAxisAlignment.start,
                        children: [
                          pw.Text(
                            entry.beschreibung,
                            style: const pw.TextStyle(fontSize: 10),
                          ),
                          pw.Text(
                            '${entry.anwender} • ${entry.zeitstempel.toString().split(' ')[0]}',
                            style: const pw.TextStyle(
                              fontSize: 8,
                              color: PdfColors.grey600,
                            ),
                          ),
                        ],
                      ),
                    ),
                  ],
                ),
              ),
            ),
      ],
    );
  }

  static pw.Widget _buildDataRow(
    String label,
    String value, {
    pw.TextStyle? valueStyle,
  }) {
    return pw.Padding(
      padding: const pw.EdgeInsets.symmetric(vertical: 2),
      child: pw.Row(
        crossAxisAlignment: pw.CrossAxisAlignment.start,
        children: [
          pw.Container(
            width: 140,
            child: pw.Text(
              '$label:',
              style: pw.TextStyle(fontWeight: pw.FontWeight.bold, fontSize: 11),
            ),
          ),
          pw.Expanded(
            child: pw.Text(
              value,
              style: valueStyle ?? const pw.TextStyle(fontSize: 11),
            ),
          ),
        ],
      ),
    );
  }

  static pw.Widget _buildFooter() {
    return pw.Container(
      padding: const pw.EdgeInsets.all(8),
      decoration: const pw.BoxDecoration(
        border: pw.Border(top: pw.BorderSide(color: PdfColors.grey300)),
      ),
      child: pw.Row(
        mainAxisAlignment: pw.MainAxisAlignment.spaceBetween,
        children: [
          pw.Text(
            'Bewerbungs-Viewer • Automatisch generiert',
            style: const pw.TextStyle(fontSize: 8, color: PdfColors.grey600),
          ),
          pw.Text(
            'Seite 1 von 1',
            style: const pw.TextStyle(fontSize: 8, color: PdfColors.grey600),
          ),
        ],
      ),
    );
  }

  static String _formatRating(int rating) {
    if (rating == 0) return '-';
    return '$rating/5';
  }

  static String _getStatusText(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return 'Ungültig';
      case ProcessingStatus.unbearbeitet:
        return 'Neu';
      case ProcessingStatus.geprueft:
        return 'Geprüft';
      case ProcessingStatus.eingeladen:
        return 'Eingeladen';
      case ProcessingStatus.gespraechGefuehrt:
        return 'Gespräch geführt';
      case ProcessingStatus.zusage:
        return 'Zusage';
      case ProcessingStatus.absage:
        return 'Absage';
    }
  }
}
