import 'dart:io';
import 'package:xml/xml.dart';
import 'package:aspiro/models/application.dart';
import 'package:aspiro/services/config_service.dart';

class XmlService {
  static XmlService? _instance;
  static XmlService get instance => _instance ??= XmlService._();
  XmlService._();

  Future<List<ApplicationGroup>> loadApplications() async {
    final String dirPath = ConfigService.instance.xmlDirectoryPath;
    final Directory directory = Directory(dirPath);

    if (!await directory.exists()) {
      return [];
    }

    final List<Application> applications = [];

    await for (final FileSystemEntity entity in directory.list()) {
      if (entity is File && entity.path.toLowerCase().endsWith('.xml')) {
        try {
          final Application? app = await _parseXmlFile(entity);
          if (app != null) {
            applications.add(app);
          }
        } catch (e) {
          print('Fehler beim Parsen von ${entity.path}: $e');
        }
      }
    }

    return _groupApplications(applications);
  }

  Future<Application?> _parseXmlFile(File file) async {
    try {
      final String content = await file.readAsString();
      final XmlDocument document = XmlDocument.parse(content);
      final XmlElement root = document.rootElement;

      final Map<String, String> allFields = {};

      for (final XmlElement element in root.children.whereType<XmlElement>()) {
        if (element.name.local != 'interne_bearbeitung') {
          allFields[element.name.local] = element.innerText;
        }
      }

      // Parse timestamp from root element
      DateTime? timestamp;
      final String? timestampAttr = root.getAttribute('t:timestamp');
      if (timestampAttr != null) {
        try {
          timestamp = DateTime.parse(timestampAttr);
        } catch (e) {
          print('Timestamp Parse Error: $e');
        }
      }

      final InternalProcessing internalProcessing = _parseInternalProcessing(
        root,
      );

      return Application(
        fileName: file.path.split('/').last,
        filePath: file.path,
        stellentitel: _getElementText(root, 'stellentitel'),
        vorname: _getElementText(root, 'vorname'),
        familienname: _getElementText(root, 'familienname'),
        anrede: _getElementText(root, 'anrede'),
        telefon1: _getElementText(root, 'telefon_1'),
        email: _getElementText(root, 'e_mail'),
        motivation: _getElementText(root, 'motivation'),
        faehigkeiten: _getElementText(
          root,
          'lebenslauf_faehigkeiten_und_kompetenzen',
        ),
        allFields: allFields,
        internalProcessing: internalProcessing,
        timestamp: timestamp,
      );
    } catch (e) {
      print('XML Parse Error: $e');
      return null;
    }
  }

  InternalProcessing _parseInternalProcessing(XmlElement root) {
    final XmlElement? interneElement = root.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == 'interne_bearbeitung')
        .firstOrNull;

    if (interneElement == null) {
      return InternalProcessing();
    }

    final InternalProcessing processing = InternalProcessing();

    // Gueltig
    processing.gueltig =
        _getElementText(interneElement, 'gueltig').toLowerCase() == 'true';

    // Geprueft
    final XmlElement? geprueftElement = interneElement.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == 'geprueft')
        .firstOrNull;

    if (geprueftElement != null) {
      processing.geprueftAnwender = _getElementText(
        geprueftElement,
        'anwender',
      );
      final String datumStr = _getElementText(geprueftElement, 'datum');
      if (datumStr.isNotEmpty) {
        try {
          processing.geprueftDatum = DateTime.parse(datumStr);
        } catch (e) {
          print('Datum Parse Error: $e');
        }
      }
    }

    // Eingeladen
    final XmlElement? eingeladenElement = interneElement.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == 'eingeladen')
        .firstOrNull;

    if (eingeladenElement != null) {
      final String datumStr = _getElementText(eingeladenElement, 'datum');
      if (datumStr.isNotEmpty) {
        try {
          processing.eingeladenDatum = DateTime.parse(datumStr);
        } catch (e) {
          print('Einladungsdatum Parse Error: $e');
        }
      }
      processing.eingeladenUhrzeit = _getElementText(
        eingeladenElement,
        'uhrzeit',
      );
      processing.wahrgenommen =
          _getElementText(eingeladenElement, 'wahrgenommen').toLowerCase() ==
          'true';
    }

    // Eignung
    final XmlElement? eignungElement = interneElement.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == 'eignung')
        .firstOrNull;

    if (eignungElement != null) {
      processing.ausbildungBewertung =
          int.tryParse(_getElementText(eignungElement, 'ausbildung')) ?? 0;
      processing.motivationBewertung =
          int.tryParse(_getElementText(eignungElement, 'motivation')) ?? 0;
      processing.gehaltsvorstellungenBewertung =
          int.tryParse(
            _getElementText(eignungElement, 'gehaltsvorstellungen'),
          ) ??
          0;
      processing.teamakzeptanzBewertung =
          int.tryParse(_getElementText(eignungElement, 'teamakzeptanz')) ?? 0;
      processing.gdb =
          int.tryParse(_getElementText(eignungElement, 'gdb')) ?? 0;
    }

    // Zusage
    processing.zusage =
        _getElementText(interneElement, 'zusage').toLowerCase() == 'true';

    // Dokumentation
    final XmlElement? dokumentationElement = interneElement.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == 'dokumentation')
        .firstOrNull;

    if (dokumentationElement != null) {
      final List<DocumentationEntry> dokumentationListe = [];
      for (final XmlElement entry
          in dokumentationElement.children.whereType<XmlElement>()) {
        final String typ = entry.getAttribute('typ') ?? '';
        final String anwender = entry.getAttribute('anwender') ?? '';
        final String zeitstempel = entry.getAttribute('zeitstempel') ?? '';
        final String beschreibung = entry.innerText;

        if (typ.isNotEmpty && beschreibung.isNotEmpty) {
          dokumentationListe.add(
            DocumentationEntry.fromXml(
              typ,
              beschreibung,
              anwender,
              zeitstempel,
            ),
          );
        }
      }
      processing.dokumentation = dokumentationListe;
    }

    return processing;
  }

  Future<bool> saveApplication(Application application) async {
    try {
      final File file = File(application.filePath);
      final String content = await file.readAsString();
      final XmlDocument document = XmlDocument.parse(content);
      final XmlElement root = document.rootElement;

      // Remove existing interne_bearbeitung if present
      root.children.removeWhere(
        (child) =>
            child is XmlElement && child.name.local == 'interne_bearbeitung',
      );

      // Add new interne_bearbeitung
      final XmlElement interneElement = XmlElement(
        XmlName('interne_bearbeitung'),
      );

      // Gueltig
      final XmlElement gueltigElement = XmlElement(XmlName('gueltig'));
      gueltigElement.innerText = application.internalProcessing.gueltig
          .toString();
      interneElement.children.add(gueltigElement);

      // Geprueft
      final XmlElement geprueftElement = XmlElement(XmlName('geprueft'));
      final XmlElement anwenderElement = XmlElement(XmlName('anwender'));
      anwenderElement.innerText =
          application.internalProcessing.geprueftAnwender;
      geprueftElement.children.add(anwenderElement);

      final XmlElement datumElement = XmlElement(XmlName('datum'));
      if (application.internalProcessing.geprueftDatum != null) {
        datumElement.innerText = application.internalProcessing.geprueftDatum!
            .toIso8601String()
            .split('T')[0];
      }
      geprueftElement.children.add(datumElement);
      interneElement.children.add(geprueftElement);

      // Eingeladen
      final XmlElement eingeladenElement = XmlElement(XmlName('eingeladen'));
      final XmlElement einladungDatumElement = XmlElement(XmlName('datum'));
      if (application.internalProcessing.eingeladenDatum != null) {
        einladungDatumElement.innerText = application
            .internalProcessing
            .eingeladenDatum!
            .toIso8601String()
            .split('T')[0];
      }
      eingeladenElement.children.add(einladungDatumElement);

      final XmlElement uhrzeitElement = XmlElement(XmlName('uhrzeit'));
      uhrzeitElement.innerText =
          application.internalProcessing.eingeladenUhrzeit;
      eingeladenElement.children.add(uhrzeitElement);

      final XmlElement wahrgenommenElement = XmlElement(
        XmlName('wahrgenommen'),
      );
      wahrgenommenElement.innerText = application
          .internalProcessing
          .wahrgenommen
          .toString();
      eingeladenElement.children.add(wahrgenommenElement);
      interneElement.children.add(eingeladenElement);

      // Eignung
      final XmlElement eignungElement = XmlElement(XmlName('eignung'));
      final XmlElement ausbildungElement = XmlElement(XmlName('ausbildung'));
      ausbildungElement.innerText = application
          .internalProcessing
          .ausbildungBewertung
          .toString();
      eignungElement.children.add(ausbildungElement);

      final XmlElement motivationElement = XmlElement(XmlName('motivation'));
      motivationElement.innerText = application
          .internalProcessing
          .motivationBewertung
          .toString();
      eignungElement.children.add(motivationElement);

      final XmlElement gehaltsElement = XmlElement(
        XmlName('gehaltsvorstellungen'),
      );
      gehaltsElement.innerText = application
          .internalProcessing
          .gehaltsvorstellungenBewertung
          .toString();
      eignungElement.children.add(gehaltsElement);

      final XmlElement teamElement = XmlElement(XmlName('teamakzeptanz'));
      teamElement.innerText = application
          .internalProcessing
          .teamakzeptanzBewertung
          .toString();
      eignungElement.children.add(teamElement);

      final XmlElement gdbElement = XmlElement(XmlName('gdb'));
      gdbElement.innerText = application.internalProcessing.gdb.toString();
      eignungElement.children.add(gdbElement);

      interneElement.children.add(eignungElement);

      // Zusage
      final XmlElement zusageElement = XmlElement(XmlName('zusage'));
      zusageElement.innerText = application.internalProcessing.zusage
          .toString();
      interneElement.children.add(zusageElement);

      // Dokumentation
      final XmlElement dokumentationElement = XmlElement(
        XmlName('dokumentation'),
      );
      for (final DocumentationEntry entry
          in application.internalProcessing.dokumentation) {
        final XmlElement entryElement = XmlElement(XmlName('eintrag'));
        entryElement.setAttribute('typ', entry.typ);
        entryElement.setAttribute('anwender', entry.anwender);
        entryElement.setAttribute(
          'zeitstempel',
          entry.zeitstempel.toIso8601String(),
        );
        entryElement.innerText = entry.beschreibung;
        dokumentationElement.children.add(entryElement);
      }
      interneElement.children.add(dokumentationElement);

      root.children.add(interneElement);

      // Save file
      await file.writeAsString(
        document.toXmlString(pretty: true, indent: '    '),
      );
      return true;
    } catch (e) {
      print('Save Error: $e');
      return false;
    }
  }

  Future<bool> deleteApplication(Application application, String reason) async {
    try {
      // Add deletion documentation before deleting
      final String currentUser = await getCurrentUser();
      final DocumentationEntry deletionEntry = DocumentationEntry(
        typ: 'LÖSCHUNG',
        beschreibung: 'Bewerbung gelöscht. Grund: $reason',
        anwender: currentUser,
        zeitstempel: DateTime.now(),
      );

      final List<DocumentationEntry> updatedDokumentation = [
        ...application.internalProcessing.dokumentation,
        deletionEntry,
      ];

      final InternalProcessing updatedProcessing = InternalProcessing(
        gueltig: application.internalProcessing.gueltig,
        geprueftAnwender: application.internalProcessing.geprueftAnwender,
        geprueftDatum: application.internalProcessing.geprueftDatum,
        eingeladenDatum: application.internalProcessing.eingeladenDatum,
        eingeladenUhrzeit: application.internalProcessing.eingeladenUhrzeit,
        wahrgenommen: application.internalProcessing.wahrgenommen,
        ausbildungBewertung: application.internalProcessing.ausbildungBewertung,
        motivationBewertung: application.internalProcessing.motivationBewertung,
        gehaltsvorstellungenBewertung:
            application.internalProcessing.gehaltsvorstellungenBewertung,
        teamakzeptanzBewertung:
            application.internalProcessing.teamakzeptanzBewertung,
        gdb: application.internalProcessing.gdb,
        zusage: application.internalProcessing.zusage,
        dokumentation: updatedDokumentation,
      );

      final Application updatedApplication = application.copyWith(
        internalProcessing: updatedProcessing,
      );

      // Save the documentation before deleting
      await saveApplication(updatedApplication);

      // Now delete the file
      final File file = File(application.filePath);
      await file.delete();
      return true;
    } catch (e) {
      print('Delete Error: $e');
      return false;
    }
  }

  Future<bool> addDocumentationEntry(
    Application application,
    String typ,
    String beschreibung,
  ) async {
    try {
      final String currentUser = await getCurrentUser();
      final DocumentationEntry newEntry = DocumentationEntry(
        typ: typ,
        beschreibung: beschreibung,
        anwender: currentUser,
        zeitstempel: DateTime.now(),
      );

      final List<DocumentationEntry> updatedDokumentation = [
        ...application.internalProcessing.dokumentation,
        newEntry,
      ];

      final InternalProcessing updatedProcessing = InternalProcessing(
        gueltig: application.internalProcessing.gueltig,
        geprueftAnwender: application.internalProcessing.geprueftAnwender,
        geprueftDatum: application.internalProcessing.geprueftDatum,
        eingeladenDatum: application.internalProcessing.eingeladenDatum,
        eingeladenUhrzeit: application.internalProcessing.eingeladenUhrzeit,
        wahrgenommen: application.internalProcessing.wahrgenommen,
        ausbildungBewertung: application.internalProcessing.ausbildungBewertung,
        motivationBewertung: application.internalProcessing.motivationBewertung,
        gehaltsvorstellungenBewertung:
            application.internalProcessing.gehaltsvorstellungenBewertung,
        teamakzeptanzBewertung:
            application.internalProcessing.teamakzeptanzBewertung,
        gdb: application.internalProcessing.gdb,
        zusage: application.internalProcessing.zusage,
        dokumentation: updatedDokumentation,
      );

      final Application updatedApplication = application.copyWith(
        internalProcessing: updatedProcessing,
      );

      return await saveApplication(updatedApplication);
    } catch (e) {
      print('Documentation Error: $e');
      return false;
    }
  }

  Future<String> getCurrentUser() async {
    try {
      final ProcessResult result = await Process.run('whoami', []);
      return result.stdout.toString().trim();
    } catch (e) {
      return Platform.environment['USER'] ?? 'Unbekannt';
    }
  }

  String _getElementText(XmlElement parent, String tagName) {
    final XmlElement? element = parent.children
        .whereType<XmlElement>()
        .where((e) => e.name.local == tagName)
        .firstOrNull;
    return element?.innerText ?? '';
  }

  List<ApplicationGroup> _groupApplications(List<Application> applications) {
    final Map<String, List<Application>> grouped = {};

    for (final Application app in applications) {
      final String key = app.stellentitel.isEmpty
          ? 'Unbekannte Stelle'
          : app.stellentitel;
      grouped.putIfAbsent(key, () => []).add(app);
    }

    return grouped.entries
        .map(
          (entry) => ApplicationGroup(
            stellentitel: entry.key,
            applications: entry.value,
          ),
        )
        .toList()
      ..sort((a, b) => a.stellentitel.compareTo(b.stellentitel));
  }
}
