import 'package:flutter/material.dart';
import 'package:aspiro/models/application.dart';
import 'package:aspiro/services/xml_service.dart';
import 'package:aspiro/services/config_service.dart';
import 'package:aspiro/services/pdf_template.dart';

class XmlViewer extends StatefulWidget {
  final Application? application;
  final List<ApplicationGroup> applicationGroups;
  final Function(Application)? onApplicationUpdated;
  final Function()? onApplicationDeleted;
  final Function(Application)? onApplicationSelected;

  const XmlViewer({
    super.key,
    this.application,
    required this.applicationGroups,
    this.onApplicationUpdated,
    this.onApplicationDeleted,
    this.onApplicationSelected,
  });

  @override
  State<XmlViewer> createState() => _XmlViewerState();
}

class _XmlViewerState extends State<XmlViewer> {
  bool _isEditing = false;
  InternalProcessing? _editingProcessing;
  final Map<String, TextEditingController> _controllers = {};

  @override
  void initState() {
    super.initState();
    _initializeControllers();
  }

  @override
  void didUpdateWidget(XmlViewer oldWidget) {
    super.didUpdateWidget(oldWidget);
    if (oldWidget.application != widget.application) {
      _initializeControllers();
      setState(() => _isEditing = false);
    }
  }

  void _initializeControllers() {
    _controllers.clear();
    if (widget.application != null) {
      _editingProcessing = InternalProcessing(
        gueltig: widget.application!.internalProcessing.gueltig,
        geprueftAnwender:
            widget.application!.internalProcessing.geprueftAnwender,
        geprueftDatum: widget.application!.internalProcessing.geprueftDatum,
        eingeladenDatum: widget.application!.internalProcessing.eingeladenDatum,
        eingeladenUhrzeit:
            widget.application!.internalProcessing.eingeladenUhrzeit,
        wahrgenommen: widget.application!.internalProcessing.wahrgenommen,
        ausbildungBewertung:
            widget.application!.internalProcessing.ausbildungBewertung,
        motivationBewertung:
            widget.application!.internalProcessing.motivationBewertung,
        gehaltsvorstellungenBewertung: widget
            .application!
            .internalProcessing
            .gehaltsvorstellungenBewertung,
        teamakzeptanzBewertung:
            widget.application!.internalProcessing.teamakzeptanzBewertung,
        gdb: widget.application!.internalProcessing.gdb,
        zusage: widget.application!.internalProcessing.zusage,
        dokumentation: List.from(
          widget.application!.internalProcessing.dokumentation,
        ),
      );

      _controllers['anwender'] = TextEditingController(
        text: _editingProcessing!.geprueftAnwender,
      );
      _controllers['uhrzeit'] = TextEditingController(
        text: _editingProcessing!.eingeladenUhrzeit,
      );
      _controllers['gdb'] = TextEditingController(
        text: _editingProcessing!.gdb > 0
            ? _editingProcessing!.gdb.toString()
            : '',
      );
    }
  }

  @override
  void dispose() {
    for (var controller in _controllers.values) {
      controller.dispose();
    }
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    if (widget.application == null) {
      return _buildUpcomingAppointmentsView(context);
    }

    return Padding(
      padding: const EdgeInsets.all(24),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          _buildHeader(context),
          const SizedBox(height: 24),
          Expanded(
            child: Row(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Expanded(flex: 2, child: _buildDetailsCard(context)),
                const SizedBox(width: 16),
                Expanded(flex: 1, child: _buildProcessingCard(context)),
              ],
            ),
          ),
        ],
      ),
    );
  }

  List<Application> _getInvitedApplicants() {
    final List<Application> allApps = widget.applicationGroups
        .expand((group) => group.applications)
        .toList();

    final invitedApps = allApps.where((app) {
      final invitedDate = app.internalProcessing.eingeladenDatum;
      // Nur anstehende Termine anzeigen (inkl. heute)
      return invitedDate != null &&
          invitedDate.isAfter(DateTime.now().subtract(const Duration(days: 1)));
    }).toList();

    // Nach Stellentitel, dann nach Datum sortieren
    invitedApps.sort((a, b) {
      int compare = a.stellentitel.compareTo(b.stellentitel);
      if (compare == 0) {
        final dateA = a.internalProcessing.eingeladenDatum;
        final dateB = b.internalProcessing.eingeladenDatum;
        if (dateA != null && dateB != null) {
          return dateA.compareTo(dateB);
        }
      }
      return compare;
    });

    return invitedApps;
  }

  Widget _buildUpcomingAppointmentsView(BuildContext context) {
    final List<Application> upcomingAppointments = _getInvitedApplicants();

    if (upcomingAppointments.isEmpty) {
      return const Center(
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(Icons.event_busy_outlined, size: 64, color: Colors.grey),
            SizedBox(height: 16),
            Text(
              'Keine anstehenden Termine',
              style: TextStyle(fontSize: 18, color: Colors.grey),
            ),
            SizedBox(height: 8),
            Text(
              'Wählen Sie links eine Bewerbung aus, um Details anzuzeigen.',
              style: TextStyle(color: Colors.grey),
            ),
          ],
        ),
      );
    }

    return Padding(
      padding: const EdgeInsets.all(24),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(
            'Anstehende Termine',
            style: Theme.of(
              context,
            ).textTheme.headlineSmall?.copyWith(fontWeight: FontWeight.bold),
          ),
          const SizedBox(height: 16),
          Expanded(
            child: ListView.builder(
              itemCount: upcomingAppointments.length,
              itemBuilder: (context, index) {
                final app = upcomingAppointments[index];
                return _buildAppointmentTile(context, app);
              },
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildAppointmentTile(BuildContext context, Application app) {
    final date = app.internalProcessing.eingeladenDatum!;
    final time = app.internalProcessing.eingeladenUhrzeit;
    final dateString = '${date.day}.${date.month}.${date.year}';

    return Card(
      margin: const EdgeInsets.only(bottom: 12),
      child: ListTile(
        onTap: () => widget.onApplicationSelected?.call(app),
        leading: CircleAvatar(
          backgroundColor: Theme.of(context).colorScheme.primaryContainer,
          child: Icon(
            Icons.event,
            color: Theme.of(context).colorScheme.onPrimaryContainer,
          ),
        ),
        title: Text(
          app.displayName,
          style: const TextStyle(fontWeight: FontWeight.bold),
        ),
        subtitle: Text(app.stellentitel),
        trailing: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          crossAxisAlignment: CrossAxisAlignment.end,
          children: [
            Text(
              dateString,
              style: const TextStyle(fontWeight: FontWeight.w600),
            ),
            if (time.isNotEmpty)
              Text(time, style: Theme.of(context).textTheme.bodySmall),
          ],
        ),
      ),
    );
  }

  Widget _buildHeader(BuildContext context) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Row(
          children: [
            CircleAvatar(
              radius: 30,
              backgroundColor: Theme.of(context).colorScheme.primary,
              child: Text(
                widget.application!.vorname.isNotEmpty
                    ? widget.application!.vorname[0].toUpperCase()
                    : '?',
                style: TextStyle(
                  color: Theme.of(context).colorScheme.onPrimary,
                  fontSize: 24,
                  fontWeight: FontWeight.bold,
                ),
              ),
            ),
            const SizedBox(width: 16),
            Expanded(
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    widget.application!.displayName,
                    style: Theme.of(context).textTheme.headlineSmall?.copyWith(
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  const SizedBox(height: 4),
                  Text(
                    'Bewerbung für: ${widget.application!.stellentitel}',
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      color: Theme.of(context).colorScheme.primary,
                    ),
                  ),
                  const SizedBox(height: 4),
                  Text(
                    'Datei: ${widget.application!.fileName}',
                    style: Theme.of(
                      context,
                    ).textTheme.bodyMedium?.copyWith(color: Colors.grey[600]),
                  ),
                ],
              ),
            ),
            _buildEditToggle(context),
          ],
        ),
      ),
    );
  }

  Widget _buildEditToggle(BuildContext context) {
    final bool canDelete =
        widget.application?.canBeDeleted(
          ConfigService.instance.deleteAfterMonths,
        ) ??
        false;

    return Wrap(
      spacing: 8,
      runSpacing: 8,
      children: [
        // PDF Export Button
        ElevatedButton.icon(
          onPressed: () => _exportToPdf(context),
          icon: const Icon(Icons.picture_as_pdf, size: 16),
          label: const Text('PDF'),
          style: ElevatedButton.styleFrom(
            foregroundColor: Colors.blue[700],
            padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
          ),
        ),

        if (canDelete && !_isEditing)
          ElevatedButton.icon(
            onPressed: () => _showDeleteDialog(context),
            icon: const Icon(Icons.delete, color: Colors.red, size: 16),
            label: const Text('Löschen'),
            style: ElevatedButton.styleFrom(
              foregroundColor: Colors.red,
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
            ),
          ),

        if (_isEditing) ...[
          TextButton(
            onPressed: _cancelEditing,
            style: TextButton.styleFrom(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
            ),
            child: const Text('Abbrechen'),
          ),
          ElevatedButton(
            onPressed: _saveChanges,
            style: ElevatedButton.styleFrom(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
            ),
            child: const Text('Speichern'),
          ),
        ] else
          ElevatedButton.icon(
            onPressed: _startEditing,
            icon: const Icon(Icons.edit, size: 16),
            label: const Text('Bearbeiten'),
            style: ElevatedButton.styleFrom(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
            ),
          ),
      ],
    );
  }

  Widget _buildDetailsCard(BuildContext context) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Bewerbungsdetails',
              style: Theme.of(
                context,
              ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 16),
            Expanded(
              child: ListView(
                children: widget.application!.allFields.entries
                    .map(
                      (entry) =>
                          _buildDetailRow(context, entry.key, entry.value),
                    )
                    .toList(),
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildProcessingCard(BuildContext context) {
    return Column(
      children: [
        Expanded(
          flex: 2,
          child: Card(
            child: Padding(
              padding: const EdgeInsets.all(20),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Text(
                        'Interne Bearbeitung',
                        style: Theme.of(context).textTheme.titleLarge?.copyWith(
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                      const Spacer(),
                      _buildStatusChip(context),
                    ],
                  ),
                  const SizedBox(height: 16),
                  Expanded(
                    child: _isEditing
                        ? _buildEditingForm(context)
                        : _buildProcessingView(context),
                  ),
                ],
              ),
            ),
          ),
        ),
        if (widget
            .application!
            .internalProcessing
            .dokumentation
            .isNotEmpty) ...[
          const SizedBox(height: 16),
          Expanded(flex: 1, child: _buildDocumentationCard(context)),
        ],
      ],
    );
  }

  Widget _buildStatusChip(BuildContext context) {
    final ProcessingStatus status =
        widget.application!.internalProcessing.status;
    return Container(
      padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
      decoration: BoxDecoration(
        color: _getStatusColor(status),
        borderRadius: BorderRadius.circular(16),
      ),
      child: Text(
        _getStatusText(status),
        style: TextStyle(
          color: _getStatusTextColor(status),
          fontSize: 12,
          fontWeight: FontWeight.w600,
        ),
      ),
    );
  }

  Widget _buildProcessingView(BuildContext context) {
    final processing = widget.application!.internalProcessing;

    return ListView(
      children: [
        _buildProcessingSection('Gültigkeit', [
          _buildProcessingRow('Gültig', processing.gueltig ? 'Ja' : 'Nein'),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Prüfung', [
          _buildProcessingRow(
            'Geprüft von',
            processing.geprueftAnwender.isEmpty
                ? '-'
                : processing.geprueftAnwender,
          ),
          _buildProcessingRow(
            'Geprüft am',
            processing.geprueftDatum?.toLocal().toString().split(' ')[0] ?? '-',
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Einladung', [
          _buildProcessingRow(
            'Datum',
            processing.eingeladenDatum?.toLocal().toString().split(' ')[0] ??
                '-',
          ),
          _buildProcessingRow(
            'Uhrzeit',
            processing.eingeladenUhrzeit.isEmpty
                ? '-'
                : processing.eingeladenUhrzeit,
          ),
          _buildProcessingRow(
            'Wahrgenommen',
            processing.wahrgenommen ? 'Ja' : 'Nein',
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Bewertung', [
          _buildProcessingRow(
            'Ausbildung',
            _buildRatingDisplay(processing.ausbildungBewertung),
          ),
          _buildProcessingRow(
            'Motivation',
            _buildRatingDisplay(processing.motivationBewertung),
          ),
          _buildProcessingRow(
            'Gehaltsvorstellungen',
            _buildRatingDisplay(processing.gehaltsvorstellungenBewertung),
          ),
          _buildProcessingRow(
            'Teamakzeptanz',
            _buildRatingDisplay(processing.teamakzeptanzBewertung),
          ),
          _buildProcessingRow(
            'GdB',
            processing.gdb > 0 ? '${processing.gdb}%' : '-',
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Entscheidung', [
          _buildProcessingRow('Zusage', processing.zusage ? 'Ja' : 'Nein'),
        ]),
      ],
    );
  }

  Widget _buildRatingDisplay(int rating) {
    if (rating == 0) return const Text('-');

    return Row(
      mainAxisSize: MainAxisSize.min,
      children: List.generate(
        5,
        (index) => Icon(
          index < rating ? Icons.star : Icons.star_border,
          size: 16,
          color: Colors.amber,
        ),
      ),
    );
  }

  Widget _buildEditingForm(BuildContext context) {
    return ListView(
      children: [
        _buildProcessingSection('Gültigkeit', [
          CheckboxListTile(
            title: const Text('Bewerbung ist gültig'),
            value: _editingProcessing!.gueltig,
            onChanged: (value) =>
                setState(() => _editingProcessing!.gueltig = value ?? false),
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Prüfung', [
          TextFormField(
            controller: _controllers['anwender'],
            decoration: const InputDecoration(
              labelText: 'Geprüft von',
              border: OutlineInputBorder(),
            ),
            onChanged: (value) => _editingProcessing!.geprueftAnwender = value,
          ),
          const SizedBox(height: 8),
          _buildDatePicker(
            label: 'Prüfungsdatum',
            date: _editingProcessing!.geprueftDatum,
            onDateSelected: (date) =>
                setState(() => _editingProcessing!.geprueftDatum = date),
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Einladung', [
          _buildDatePicker(
            label: 'Einladungsdatum',
            date: _editingProcessing!.eingeladenDatum,
            onDateSelected: (date) =>
                setState(() => _editingProcessing!.eingeladenDatum = date),
          ),
          const SizedBox(height: 8),
          TextFormField(
            controller: _controllers['uhrzeit'],
            decoration: const InputDecoration(
              labelText: 'Uhrzeit (z.B. 14:30)',
              border: OutlineInputBorder(),
            ),
            onChanged: (value) => _editingProcessing!.eingeladenUhrzeit = value,
          ),
          const SizedBox(height: 8),
          CheckboxListTile(
            title: const Text('Termin wahrgenommen'),
            value: _editingProcessing!.wahrgenommen,
            onChanged: (value) => setState(
              () => _editingProcessing!.wahrgenommen = value ?? false,
            ),
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Bewertung (1-5 Sterne)', [
          _buildRatingField(
            'Ausbildung',
            _editingProcessing!.ausbildungBewertung,
            (rating) => setState(
              () => _editingProcessing!.ausbildungBewertung = rating,
            ),
          ),
          _buildRatingField(
            'Motivation',
            _editingProcessing!.motivationBewertung,
            (rating) => setState(
              () => _editingProcessing!.motivationBewertung = rating,
            ),
          ),
          _buildRatingField(
            'Gehaltsvorstellungen',
            _editingProcessing!.gehaltsvorstellungenBewertung,
            (rating) => setState(
              () => _editingProcessing!.gehaltsvorstellungenBewertung = rating,
            ),
          ),
          _buildRatingField(
            'Teamakzeptanz',
            _editingProcessing!.teamakzeptanzBewertung,
            (rating) => setState(
              () => _editingProcessing!.teamakzeptanzBewertung = rating,
            ),
          ),
          const SizedBox(height: 8),
          TextFormField(
            controller: _controllers['gdb'],
            keyboardType: TextInputType.number,
            decoration: const InputDecoration(
              labelText: 'Grad der Behinderung (GdB) in %',
              border: OutlineInputBorder(),
              suffixText: '%',
              helperText: '0-100%',
            ),
            onChanged: (value) {
              final int? gdbValue = int.tryParse(value);
              if (gdbValue != null && gdbValue >= 0 && gdbValue <= 100) {
                _editingProcessing!.gdb = gdbValue;
              } else if (value.isEmpty) {
                _editingProcessing!.gdb = 0;
              }
            },
          ),
        ]),
        const SizedBox(height: 16),
        _buildProcessingSection('Entscheidung', [
          CheckboxListTile(
            title: const Text('Zusage erteilen'),
            value: _editingProcessing!.zusage,
            onChanged: (value) =>
                setState(() => _editingProcessing!.zusage = value ?? false),
          ),
        ]),
      ],
    );
  }

  Widget _buildDatePicker({
    required String label,
    required DateTime? date,
    required Function(DateTime?) onDateSelected,
  }) {
    return InkWell(
      onTap: () async {
        final DateTime? picked = await showDatePicker(
          context: context,
          initialDate: date ?? DateTime.now(),
          firstDate: DateTime(2020),
          lastDate: DateTime(2030),
        );
        onDateSelected(picked);
      },
      child: Container(
        padding: const EdgeInsets.all(16),
        decoration: BoxDecoration(
          border: Border.all(color: Colors.grey[400]!),
          borderRadius: BorderRadius.circular(4),
        ),
        child: Row(
          children: [
            Expanded(
              child: Text(
                date != null ? date.toLocal().toString().split(' ')[0] : label,
                style: TextStyle(color: date != null ? null : Colors.grey[600]),
              ),
            ),
            const Icon(Icons.calendar_today),
          ],
        ),
      ),
    );
  }

  Widget _buildRatingField(
    String label,
    int currentRating,
    Function(int) onRatingChanged,
  ) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(label, style: const TextStyle(fontWeight: FontWeight.w500)),
        const SizedBox(height: 4),
        Row(
          children: List.generate(
            5,
            (index) => GestureDetector(
              onTap: () => onRatingChanged(index + 1),
              child: Icon(
                index < currentRating ? Icons.star : Icons.star_border,
                color: Colors.amber,
                size: 32,
              ),
            ),
          ),
        ),
        const SizedBox(height: 8),
      ],
    );
  }

  Widget _buildProcessingSection(String title, List<Widget> children) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          title,
          style: const TextStyle(fontWeight: FontWeight.w600, fontSize: 14),
        ),
        const SizedBox(height: 8),
        ...children,
      ],
    );
  }

  Widget _buildProcessingRow(String label, dynamic value) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          SizedBox(
            width: 120,
            child: Text(
              label,
              style: const TextStyle(fontWeight: FontWeight.w500),
            ),
          ),
          Expanded(child: value is Widget ? value : Text(value.toString())),
        ],
      ),
    );
  }

  Widget _buildDetailRow(BuildContext context, String key, String value) {
    final String displayKey = _formatFieldName(key);
    final String displayValue = value.isEmpty ? '-' : value;

    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 8),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          SizedBox(
            width: 200,
            child: Text(
              displayKey,
              style: Theme.of(context).textTheme.bodyMedium?.copyWith(
                fontWeight: FontWeight.w600,
                color: Theme.of(context).colorScheme.primary,
              ),
            ),
          ),
          Expanded(
            child: Container(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
              decoration: BoxDecoration(
                color: Theme.of(context).colorScheme.surfaceContainerHighest,
                borderRadius: BorderRadius.circular(6),
              ),
              child: Text(
                displayValue,
                style: Theme.of(context).textTheme.bodyMedium,
              ),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildDocumentationCard(BuildContext context) {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(20),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Dokumentation',
              style: Theme.of(
                context,
              ).textTheme.titleMedium?.copyWith(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 12),
            Expanded(
              child: ListView.builder(
                itemCount:
                    widget.application!.internalProcessing.dokumentation.length,
                itemBuilder: (context, index) {
                  final DocumentationEntry entry = widget
                      .application!
                      .internalProcessing
                      .dokumentation[index];
                  return Container(
                    margin: const EdgeInsets.only(bottom: 8),
                    padding: const EdgeInsets.all(12),
                    decoration: BoxDecoration(
                      color: Theme.of(
                        context,
                      ).colorScheme.surfaceContainerHighest,
                      borderRadius: BorderRadius.circular(8),
                      border: Border(
                        left: BorderSide(
                          width: 4,
                          color: entry.typ == 'LÖSCHUNG'
                              ? Colors.red
                              : entry.typ == 'ÄNDERUNG'
                              ? Colors.blue
                              : Colors.green,
                        ),
                      ),
                    ),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Row(
                          children: [
                            Container(
                              padding: const EdgeInsets.symmetric(
                                horizontal: 6,
                                vertical: 2,
                              ),
                              decoration: BoxDecoration(
                                color: entry.typ == 'LÖSCHUNG'
                                    ? Colors.red[100]
                                    : entry.typ == 'ÄNDERUNG'
                                    ? Colors.blue[100]
                                    : Colors.green[100],
                                borderRadius: BorderRadius.circular(4),
                              ),
                              child: Text(
                                entry.typ,
                                style: TextStyle(
                                  color: entry.typ == 'LÖSCHUNG'
                                      ? Colors.red[700]
                                      : entry.typ == 'ÄNDERUNG'
                                      ? Colors.blue[700]
                                      : Colors.green[700],
                                  fontSize: 10,
                                  fontWeight: FontWeight.bold,
                                ),
                              ),
                            ),
                            const Spacer(),
                            Text(
                              entry.zeitstempel.toLocal().toString().split(
                                ' ',
                              )[0],
                              style: Theme.of(context).textTheme.bodySmall
                                  ?.copyWith(color: Colors.grey[600]),
                            ),
                          ],
                        ),
                        const SizedBox(height: 4),
                        Text(
                          entry.beschreibung,
                          style: Theme.of(context).textTheme.bodySmall,
                        ),
                        const SizedBox(height: 2),
                        Text(
                          'von ${entry.anwender}',
                          style: Theme.of(context).textTheme.bodySmall
                              ?.copyWith(
                                color: Colors.grey[600],
                                fontStyle: FontStyle.italic,
                              ),
                        ),
                      ],
                    ),
                  );
                },
              ),
            ),
          ],
        ),
      ),
    );
  }

  Future<void> _showDeleteDialog(BuildContext context) async {
    final TextEditingController reasonController = TextEditingController();

    final bool? shouldDelete = await showDialog<bool>(
      context: context,
      builder: (context) => AlertDialog(
        title: const Text('Bewerbung löschen'),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Sind Sie sicher, dass Sie die Bewerbung von ${widget.application!.displayName} löschen möchten?',
              style: Theme.of(context).textTheme.bodyMedium,
            ),
            const SizedBox(height: 16),
            Text(
              'Diese Aktion kann nicht rückgängig gemacht werden.',
              style: Theme.of(context).textTheme.bodySmall?.copyWith(
                color: Colors.red[700],
                fontWeight: FontWeight.w600,
              ),
            ),
            const SizedBox(height: 16),
            TextField(
              controller: reasonController,
              decoration: const InputDecoration(
                labelText: 'Grund der Löschung (erforderlich)',
                border: OutlineInputBorder(),
                helperText:
                    'z.B. Datenschutzfrist abgelaufen, auf Wunsch des Bewerbers',
              ),
              maxLines: 3,
            ),
          ],
        ),
        actions: [
          TextButton(
            onPressed: () => Navigator.of(context).pop(false),
            child: const Text('Abbrechen'),
          ),
          ElevatedButton(
            onPressed: () {
              if (reasonController.text.trim().isNotEmpty) {
                Navigator.of(context).pop(true);
              }
            },
            style: ElevatedButton.styleFrom(
              backgroundColor: Colors.red,
              foregroundColor: Colors.white,
            ),
            child: const Text('Löschen'),
          ),
        ],
      ),
    );

    if (shouldDelete == true && reasonController.text.trim().isNotEmpty) {
      await _deleteApplication(reasonController.text.trim());
    }
  }

  Future<void> _deleteApplication(String reason) async {
    final bool success = await XmlService.instance.deleteApplication(
      widget.application!,
      reason,
    );

    if (success) {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Bewerbung wurde gelöscht')),
        );
        widget.onApplicationDeleted?.call();
      }
    } else {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Text('Fehler beim Löschen der Bewerbung'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  Future<void> _startEditing() async {
    // Initialize with current user and date if not set
    if (_editingProcessing!.geprueftAnwender.isEmpty) {
      final String currentUser = await XmlService.instance.getCurrentUser();
      _editingProcessing!.geprueftAnwender = currentUser;
      _controllers['anwender']!.text = currentUser;
    }

    if (_editingProcessing!.geprueftDatum == null) {
      _editingProcessing!.geprueftDatum = DateTime.now();
    }

    setState(() => _isEditing = true);
  }

  void _cancelEditing() {
    _initializeControllers();
    setState(() => _isEditing = false);
  }

  Future<void> _exportToPdf(BuildContext context) async {
    if (widget.application == null) return;

    try {
      // Loading indicator anzeigen
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Row(
              children: [
                SizedBox(
                  width: 16,
                  height: 16,
                  child: CircularProgressIndicator(strokeWidth: 2),
                ),
                SizedBox(width: 12),
                Text('PDF wird erstellt...'),
              ],
            ),
            duration: Duration(seconds: 2),
          ),
        );
      }

      // PDF erstellen
      final bool success = await PdfTemplate.exportBewerbung(
        widget.application!,
      );

      if (mounted) {
        ScaffoldMessenger.of(context).hideCurrentSnackBar();

        if (success) {
          // Dateiname anzeigen
          final String fileName =
              'bewerbung_${widget.application!.vorname.toLowerCase()}_${widget.application!.familienname.toLowerCase()}_${DateTime.now().toString().split(' ')[0]}.pdf'
                  .replaceAll(' ', '_')
                  .replaceAll('ä', 'ae')
                  .replaceAll('ö', 'oe')
                  .replaceAll('ü', 'ue')
                  .replaceAll('ß', 'ss');

          ScaffoldMessenger.of(context).showSnackBar(
            SnackBar(
              content: Row(
                children: [
                  const Icon(Icons.check_circle, color: Colors.green),
                  const SizedBox(width: 8),
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        const Text('PDF erfolgreich erstellt'),
                        Text(
                          fileName,
                          style: const TextStyle(
                            fontSize: 12,
                            color: Colors.white70,
                          ),
                        ),
                      ],
                    ),
                  ),
                ],
              ),
              backgroundColor: Colors.green[600],
              duration: const Duration(seconds: 4),
            ),
          );
        } else {
          ScaffoldMessenger.of(context).showSnackBar(
            const SnackBar(
              content: Row(
                children: [
                  Icon(Icons.error, color: Colors.white),
                  SizedBox(width: 8),
                  Text('Fehler beim PDF-Export'),
                ],
              ),
              backgroundColor: Colors.red,
            ),
          );
        }
      }
    } catch (e) {
      if (mounted) {
        ScaffoldMessenger.of(context).hideCurrentSnackBar();
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(
            content: Text('PDF-Export fehlgeschlagen: $e'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  Future<void> _saveChanges() async {
    // Add documentation entry for changes
    final bool hasChanges =
        _editingProcessing!.gueltig !=
            widget.application!.internalProcessing.gueltig ||
        _editingProcessing!.geprueftAnwender !=
            widget.application!.internalProcessing.geprueftAnwender ||
        _editingProcessing!.geprueftDatum !=
            widget.application!.internalProcessing.geprueftDatum ||
        _editingProcessing!.eingeladenDatum !=
            widget.application!.internalProcessing.eingeladenDatum ||
        _editingProcessing!.eingeladenUhrzeit !=
            widget.application!.internalProcessing.eingeladenUhrzeit ||
        _editingProcessing!.wahrgenommen !=
            widget.application!.internalProcessing.wahrgenommen ||
        _editingProcessing!.ausbildungBewertung !=
            widget.application!.internalProcessing.ausbildungBewertung ||
        _editingProcessing!.motivationBewertung !=
            widget.application!.internalProcessing.motivationBewertung ||
        _editingProcessing!.gehaltsvorstellungenBewertung !=
            widget
                .application!
                .internalProcessing
                .gehaltsvorstellungenBewertung ||
        _editingProcessing!.teamakzeptanzBewertung !=
            widget.application!.internalProcessing.teamakzeptanzBewertung ||
        _editingProcessing!.gdb != widget.application!.internalProcessing.gdb ||
        _editingProcessing!.zusage !=
            widget.application!.internalProcessing.zusage;

    if (hasChanges) {
      final String currentUser = await XmlService.instance.getCurrentUser();
      final DocumentationEntry changeEntry = DocumentationEntry(
        typ: 'ÄNDERUNG',
        beschreibung: 'Bewerbungsdaten wurden bearbeitet und aktualisiert',
        anwender: currentUser,
        zeitstempel: DateTime.now(),
      );

      _editingProcessing!.dokumentation = [
        ...widget.application!.internalProcessing.dokumentation,
        changeEntry,
      ];
    }

    final Application updatedApplication = widget.application!.copyWith(
      internalProcessing: _editingProcessing!,
    );

    final bool success = await XmlService.instance.saveApplication(
      updatedApplication,
    );

    if (success) {
      setState(() => _isEditing = false);
      widget.onApplicationUpdated?.call(updatedApplication);

      if (mounted) {
        ScaffoldMessenger.of(
          context,
        ).showSnackBar(const SnackBar(content: Text('Änderungen gespeichert')));
      }
    } else {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Text('Fehler beim Speichern'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  String _formatFieldName(String fieldName) {
    final Map<String, String> fieldMappings = {
      'stellentitel': 'Stellentitel',
      'vorname': 'Vorname',
      'familienname': 'Familienname',
      'anrede': 'Anrede',
      'strasse': 'Straße',
      'postleitzahl': 'Postleitzahl',
      'ort': 'Ort',
      'telefon_1': 'Telefon',
      'e_mail': 'E-Mail',
      'geburtsdatum': 'Geburtsdatum',
      'motivation': 'Motivation',
      'lebenslauf_faehigkeiten_und_kompetenzen': 'Fähigkeiten & Kompetenzen',
      'moegliches_eintrittsdatum': 'Mögliches Eintrittsdatum',
      'gewuenschte_arbeitszeit': 'Gewünschte Arbeitszeit',
      'datenschutzerklaerung_akzeptieren': 'Datenschutzerklärung akzeptiert',
      'einwilligung_zur_weiterverarbeitung':
          'Einwilligung zur Weiterverarbeitung',
    };

    return fieldMappings[fieldName] ??
        fieldName
            .replaceAll('_', ' ')
            .split(' ')
            .map(
              (word) =>
                  word.isEmpty ? '' : word[0].toUpperCase() + word.substring(1),
            )
            .join(' ');
  }

  Color _getStatusColor(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return Colors.red[400]!;
      case ProcessingStatus.unbearbeitet:
        return Colors.grey[400]!;
      case ProcessingStatus.geprueft:
        return Colors.blue[400]!;
      case ProcessingStatus.eingeladen:
        return Colors.orange[400]!;
      case ProcessingStatus.gespraechGefuehrt:
        return Colors.purple[400]!;
      case ProcessingStatus.zusage:
        return Colors.green[400]!;
      case ProcessingStatus.absage:
        return Colors.red[300]!;
    }
  }

  Color _getStatusTextColor(ProcessingStatus status) {
    return Colors.white;
  }

  String _getStatusText(ProcessingStatus status) {
    switch (status) {
      case ProcessingStatus.ungueltig:
        return 'Ungültig';
      case ProcessingStatus.unbearbeitet:
        return 'Neu';
      case ProcessingStatus.geprueft:
        return 'Geprüft';
      case ProcessingStatus.eingeladen:
        return 'Eingeladen';
      case ProcessingStatus.gespraechGefuehrt:
        return 'Gespräch geführt';
      case ProcessingStatus.zusage:
        return 'Zusage';
      case ProcessingStatus.absage:
        return 'Absage';
    }
  }
}
