#!/usr/bin/python3

# Initalizes KDE bookmark file and makes sure the gtk bookmarks keep up

import xml.etree.ElementTree as ElementTree
import os
import shutil
import sys

from booksynclib import Bookmark,Update

ElementTree.register_namespace("bookmark", "http://www.freedesktop.org/standards/desktop-bookmarks")

# Bookmark class

def parse_bookmarks_from_root(root):
	present_bookmarks = []
	n = 1
	for child in root:
		if child.tag == "bookmark":
			if "href" in child.attrib:
				is_system = False
				include_in_gtk = None
				name = "Irgentwo"
				icon = None
				identifier = None
				for elem in child:
					if elem.tag == "title":
						name = elem.text
					if elem.tag == "info":
						for info in elem:
							if info.tag == "metadata" and "owner" in info.attrib:
								if info.attrib["owner"] == "http://freedesktop.org":
									for metadata in info:
										if metadata.tag == "{http://www.freedesktop.org/standards/desktop-bookmarks}icon":
											icon = metadata.attrib["name"]
								if info.attrib["owner"] == "http://www.kde.org":
									for metadata in info:
										if metadata.tag == "isSystemItem":
											is_system = metadata.text.lower() == "true"
										if metadata.tag == "ID":
											identifier = metadata.text
								if info.attrib["owner"] == "http://treuchtlingen.de":
									for metadata in info:
										if metadata.tag == "include_in_gtk":
											include_in_gtk = metadata.text.lower() == "true"
				present_bookmarks.append(Bookmark(name, child.attrib["href"], icon, is_system, identifier, include_in_gtk, child, root, n))
				n += 1
	return present_bookmarks

def read_update_flags(root):
	update_flags = []
	for child in root:
		if child.tag == "info":
			for info in child:
				if info.tag == "metadata" and "owner" in info.attrib:
					if info.attrib["owner"] == "http://treuchtlingen.de":
						for metadata in info:
							if metadata.tag == "update_flags":
								update_flags = metadata.text.split(" ")
	return update_flags

def find_write_or_create_element(parent, element_type, default_attrib={}, xpath=None, text=None, set_text=False):
	if xpath is None:
		xpath = "./"+element_type
	node = parent.find(xpath)
	if (set_text and text is None):
		if not (node is None):
			parent.remove(node)
	else:
		if node is None:
			node = ElementTree.Element(element_type, attrib=default_attrib)
			parent.append(node)
		if not (text is None):
			node.text = text
	return node
	
def write_update_flags(root, update_flags):
	info_node = find_write_or_create_element(root, "info")
	treuchtlingen_info_node = find_write_or_create_element(info_node, "metadata", {'owner':"http://treuchtlingen.de"}, "./metadata[@owner='http://treuchtlingen.de']")
	update_flags_node = find_write_or_create_element(treuchtlingen_info_node, "update_flags", text=(" ".join(update_flags)))

def serialize_for_gtk(bookmarks):
	gtk_bookmarks = ""
	for bookmark in bookmarks:
		if bookmark.include_in_gtk or (bookmark.include_in_gtk is None and not bookmark.is_system):
			gtk_bookmarks += bookmark.uri+" "+bookmark.name+"\n"
	return gtk_bookmarks

def run_updates(bookmarks, update_flags, updates):
	changes_made = 0
	for update in updates:
		if update.flag in update_flags:
			print("update "+str(update.flag)+" i")
		else:
			print("update "+str(update.flag)+" * // "+update.name)
			update.update_function(bookmarks)
			changes_made += 1
			print("update "+str(update.flag)+" +")
			update_flags.append(update.flag)
	return changes_made
	
def bookmark_sort_key(bookmark):
	return bookmark.n

def dump_bookmarks(bookmarks, dump_id):
	print("")
	print("dump "+dump_id+" ---")
	for bookmark in bookmarks:
		print(bookmark.n, (bookmark.uri, bookmark.name, bookmark.icon, bookmark.id, bookmark.is_system, bookmark.include_in_gtk, bookmark.deleteflag))
	print("---")
	print("")
	
######################################################
### Update functions ##############################
################################################
	
def apply_update(bookmarks, default_xml_parent):
	for bookmark in bookmarks:
		if not (bookmark.xml_parent is None):
			bookmark.xml_parent.remove(bookmark.xml_node)
		else:
			bookmark.xml_parent = default_xml_parent
	for bookmark in bookmarks:
		if not bookmark.deleteflag:
			if (bookmark.xml_node is None):
				bookmark.xml_node = ElementTree.Element("bookmark")
			bookmark.xml_node.attrib["href"] = bookmark.uri
			bookmark.xml_parent.append(bookmark.xml_node)
			title_node = find_write_or_create_element(bookmark.xml_node, "title", text=bookmark.name)
			info_node = find_write_or_create_element(bookmark.xml_node, "info")
			# get metadata nodes
			freedesktop_info_node = find_write_or_create_element(info_node, "metadata", {'owner': "http://freedesktop.org"}, "./metadata[@owner='http://freedesktop.org']")
			kde_info_node = find_write_or_create_element(info_node, "metadata", {'owner': "http://www.kde.org"}, "./metadata[@owner='http://www.kde.org']")
			treuchtlingen_info_node = find_write_or_create_element(info_node, "metadata", {'owner': "http://treuchtlingen.de"}, "./metadata[@owner='http://treuchtlingen.de']")
			# serialize include in gtk
			if (bookmark.include_in_gtk is None):
				find_write_or_create_element(treuchtlingen_info_node, "include_in_gtk", text=None, set_text=True)
			else:
				find_write_or_create_element(treuchtlingen_info_node, "include_in_gtk", text=bookmark.include_in_gtk.__str__().lower())
			
			#serialize icon information
			if bookmark.icon is None:
				find_write_or_create_element(freedesktop_info_node, "{http://www.freedesktop.org/standards/desktop-bookmarks}icon", set_text=True)
			else:
				find_write_or_create_element(freedesktop_info_node, "{http://www.freedesktop.org/standards/desktop-bookmarks}icon").attrib["name"] = bookmark.icon
			
			
			#serialize kde id information
			find_write_or_create_element(kde_info_node, "ID", text=bookmark.id, set_text=True)
			
			#serialize kde is system information
			if (bookmark.id is None):
				find_write_or_create_element(kde_info_node, "isSystemItem", text=None, set_text=False)
			else:
				find_write_or_create_element(kde_info_node, "isSystemItem", text=bookmark.is_system.__str__().lower())

######################################################
### Update descriptions ###########################
################################################

updates = []

######################################################
### Logic #########################################
################################################

from importlib.machinery import SourceFileLoader
def load_updates(path):
	bookmark_updates = SourceFileLoader("bookmark_updates",path).load_module()
	return bookmark_updates.updates
		
bookmark_xbel_path = os.getenv("HOME")+'/.local/share/user-places.xbel'
bookmark_gtk_path = os.getenv("HOME")+'/.config/gtk-3.0/bookmarks'
bookmark_update_paths = []

enable_updates = False
enable_gtk_export = False

next_is_xbel_path = False
next_is_gtk_path = False
next_is_update_path = False

is_first_arg = True

for arg in sys.argv:
	if is_first_arg:
		is_first_arg = False
	elif next_is_xbel_path:
		bookmark_xbel_path = arg
		next_is_xbel_path = False
	elif next_is_gtk_path:
		bookmark_gtk_path = arg
		next_is_gtk_path = False
	elif next_is_update_path:
		bookmark_update_paths.append(arg)
		next_is_update_path = False
	elif arg == "--enable-gtk-export":
		enable_gtk_export = True
	elif arg == "--set-gtk-path":
		enable_gtk_export = True
		next_is_gtk_path = True
	elif arg == "--apply-updates-from":
		enable_updates = True
		next_is_update_path = True
	elif arg == "--set-xbel-path":
		next_is_xbel_path = True
	else:
		print("Invalid command line argument:",arg)
		sys.exit(1)

if next_is_gtk_path or next_is_update_path or next_is_xbel_path:
	print("Last argument was incomplete!")
	sys.exit(1)

if len(bookmark_update_paths) > 0:
	print("loading updates")
	for path in bookmark_update_paths:
		print("loading update file", path)
		new_updates = load_updates(path)
		for u in new_updates:
			updates.append(u)
	

print("reading "+bookmark_xbel_path)
tree = ElementTree.parse(bookmark_xbel_path)
print("parsing bookmarks")
present_bookmarks = parse_bookmarks_from_root(tree.getroot())

dump_bookmarks(present_bookmarks, "initial")

if enable_updates:
	print("backup xbel to "+bookmark_xbel_path+".bak")
	shutil.copy2(bookmark_xbel_path, bookmark_xbel_path+".bak")
	update_flags = read_update_flags(tree.getroot())
	print("update_flags initial", update_flags)
	if run_updates(present_bookmarks, update_flags, updates) > 0:
		present_bookmarks = sorted(present_bookmarks, key=bookmark_sort_key)
		apply_update(present_bookmarks, tree.getroot())
		print("update_flags updated", update_flags)
		write_update_flags(tree.getroot(), update_flags)
		dump_bookmarks(present_bookmarks, "updated")
		ElementTree.indent(tree, "\t")
		tree.write(bookmark_xbel_path, xml_declaration=True, encoding="utf-8", method="xml")
	else:
		print("no changes neccessary")


print("")

if enable_gtk_export:
	print("backup gtk to "+bookmark_gtk_path+".bak")
	try:
		shutil.copy2(bookmark_gtk_path, bookmark_gtk_path+".bak")
	except FileNotFoundError:
		pass #file not found means we are not loosing data
	file = open(bookmark_gtk_path, "w")
	file.write(serialize_for_gtk(present_bookmarks))
	file.close()
	
