#!/bin/python3

from datetime import datetime,timedelta
from subprocess import Popen, PIPE
import argparse
import copy
import json
import time
import queue
import threading

class MonitoringResult:
	def __init__(self, name, exit_code, stdout, stderr):
		self.name = name
		self.exit_code = exit_code
		self.stdout = stdout
		self.stderr = stderr
		self.timestamp = datetime.now()

	def is_ok(self):
		return self.exit_code == 0

class MonitorWrapper:
	def __init__(self, name, command, result_queue):
		self.name = name
		self.command = command
		self.result_queue = result_queue

	def do_check(self):
		with Popen(self.command, stdout=PIPE) as proc:
			output, err = proc.communicate()
			result = MonitoringResult(
				self.name,
				proc.returncode,
				output,
				err
			)
			self.result_queue.put(result)

class CheckConfiguration:
	def __init__(self, name, type, command=[], is_required_by=None, requires=None, invert_result=False):
		self.name = name
		self.command = command
		self.type = type
		self.is_required_by = is_required_by or []
		self.requires = requires or []
		self.min_interval = timedelta(seconds=10)
		self.max_interval = timedelta(seconds=300)
		self.invert_result = invert_result
		self.trigger_edge = "both" # "rising", "falling"
		self.trigger_on_init = True
		self.report_topic = None
		self.report_label = None
		self.report_include_if = None
		self.report_destination_path = None

def read_json_to_check_configuration(name, dictionary):
	print(f"Parsing configuration for {name}: {dictionary} ...")
	
	config = CheckConfiguration(
		name,
		dictionary["type"]
	)
	if "invert_result" in dictionary:
		config.invert_result = dictionary["invert_result"]
	if "command" in dictionary:
		config.command = dictionary["command"]
	if "is_required_by" in dictionary:
		config.is_required_by = dictionary["is_required_by"]
	if "requires" in dictionary:
		config.requires = dictionary["requires"]
	if "min_interval" in dictionary:
		config.min_interval = timedelta(seconds=dictionary["min_interval"])
	if "max_interval" in dictionary:
		config.max_interval = timedelta(seconds=dictionary["max_interval"])
	if "trigger_edge" in dictionary:
		config.trigger_edge = dictionary["trigger_edge"]
	if "trigger_on_init" in dictionary:
		config.trigger_on_init = dictionary["trigger_on_init"]
	if "report_topic" in dictionary:
		config.report_topic = dictionary["report_topic"]
	if "report_label" in dictionary:
		config.report_label = dictionary["report_label"]
	if "report_include_if" in dictionary:
		config.report_include_if = dictionary["report_include_if"]
	if "report_destination_path" in dictionary:
		config.report_destination_path = dictionary["report_destination_path"]
	return config

def get_edge_type(before, after):
	if before != after:
		if after:
			return "rising"
		else:
			return "falling"
	return None

class CheckInformation:
	def __init__(self, configuration):
		self.configuration = configuration
		self.is_required_by = copy.copy(configuration.is_required_by)
		self.requires = copy.copy(configuration.requires)
		self.is_scheduled = False
		self.last_result = None # Is a MonitoringResult
		self.unmet_requirements = []
		self.interval = configuration.min_interval
		self.was_ok_on_last_update = None
		self.schedule_asap = False

	def is_active(self):
		return len(self.unmet_requirements) == 0

	def can_be_scheduled(self):
		match self.configuration.type:
			case "check" | "hook":
				return self.is_active() and not self.is_scheduled
			case _:
				return False

	def should_be_scheduled(self):
		if not self.can_be_scheduled():
			return False
		if self.configuration.type == "hook":
			return self.schedule_asap
		if self.last_result == None or self.schedule_asap:
			return True
		return datetime.now() > self.last_result.timestamp + self.interval

	def mark_scheduled(self):
		self.is_scheduled = True
		self.schedule_asap = False

	def is_ok(self):
		if self.configuration.invert_result:
			return not self.inner_is_ok()
		else:
			return self.inner_is_ok()
		
	def inner_is_ok(self):
		match self.configuration.type:
			case "check" | "hook":
				if self.last_result == None:
					return False
				return self.last_result.is_ok() and len(self.unmet_requirements) == 0
			case "and":
				return len(self.unmet_requirements) == 0
			case "or":
				return len(self.unmet_requirements) < len(self.requires)
			case "trigger":
				return True
			case _:
				return False

	def update_interval(self, changed):
		if changed:
			self.interval = self.configuration.min_interval
		else:
			d = self.interval * 1.5
			if d > self.configuration.max_interval:
				self.interval = self.configuration.max_interval
			else:
				self.interval = d
		print(f"Interval of {self.configuration.name} is now: {self.interval}")
				
	def on_report_update(self, core):
		print("==============================")
		print(f"Report: {self.configuration.name}")
		for name in self.requires:
			if name in core.service_map:
				service = core.service_map[name]
				service.print_report_structure(core)
		print("==============================")

	def is_includable_in_report(self, core):
		if self.configuration.type == "trigger":
			return False
		if self.configuration.report_include_if in core.service_map:
			service = core.service_map[self.configuration.report_include_if]
			return service.is_ok()
		return True

	def print_report_structure(self, core):
		if not self.is_includable_in_report(core):
			return
		if self.configuration.report_topic:
			label = "[ERROR]"
			if self.is_ok():
				label = "[OK]"
			print(f"# {label} {self.configuration.report_topic}")
		elif self.configuration.report_label:
			label = "[ERROR]"
			if not self.can_be_scheduled():
				label = "[WAIT]"
			elif self.is_ok():
				label = "[OK]"
			print(f"{label} {self.configuration.report_label}")
			return
		# Maybe test for a report section flag to stop early if we know, that a report
		# is not branching out further
		for name in self.requires:
			if name in core.service_map:
				service = core.service_map[name]
				service.print_report_structure(core)

	def write_report_to_file(self, core):
		if not self.configuration.report_destination_path:
			return
		struct = self.get_report_struct(core)
		with open(self.configuration.report_destination_path, "w") as f:
			f.truncate(0)
			json.dump(struct, f, indent="\t")
			f.write("\n")
			

	def get_report_struct(self, core):
		if not self.is_includable_in_report(core):
			return None
		report_base = {
			"state": "ERROR"
		}
		is_topic = False
		if self.configuration.report_topic:
			report_base["type"] = "topic"
			report_base["label"] = self.configuration.report_topic
			if self.is_ok():
				report_base["state"] = "OK"
			is_topic = True
		elif self.configuration.report_label:
			report_base["type"] = "item"
			report_base["label"] = self.configuration.report_label
			if not self.can_be_scheduled():
				report_base["state"] = "WAIT"
			elif self.is_ok():
				report_base["state"] = "OK"
			return report_base
		items = []
		for name in self.requires:
			if name in core.service_map:
				service = core.service_map[name]
				report_struct = service.get_report_struct(core)
				if type(report_struct) is list:
					for item in list:
						items.append(item)
				elif type(report_struct) is dict:
					items.append(report_struct)
		if is_topic:
			report_base["items"] = items
			return report_base
		else:
			return items
		
	
	def on_update(self, core):
		ok = self.is_ok()
		if ok != self.was_ok_on_last_update:
			print(f"State of {self.configuration.name} is now {ok} (inverted: {self.configuration.invert_result})")
		if self.configuration.type == "report":
			self.on_report_update(core)
			self.write_report_to_file(core)
			
		# Don't schedule asap if scheduling is not possible to prevent hooks from firing way too late
		if not self.can_be_scheduled():
			self.schedule_asap = False
		self.was_ok_on_last_update = ok

	def process_trigger(self, edge, is_initial_trigger, core):
		print(f"-> Received {edge} trigger on {self.configuration.name} (initial: {is_initial_trigger})")
		if is_initial_trigger and not self.configuration.trigger_on_init:
			return
		if self.configuration.type == "trigger":
			if edge and (edge == "trigger" or self.configuration.trigger_edge == edge or self.configuration.trigger_edge == "both"):
				print(f"\x1b[01mTriggered {self.configuration.name} on edge {edge}\x1b[00m")
				for service_name in self.is_required_by:
					print(f"* {service_name}")
					if service_name in core.service_map:
						service = core.service_map[service_name]
						service.process_trigger("trigger", is_initial_trigger, core)
		elif self.configuration.type == "check":
			if edge and (edge == "trigger" or self.configuration.trigger_edge == edge or self.configuration.trigger_edge == "both"):
				print(f"Scheduling check {self.configuration.name} ASAP")
				self.schedule_asap = True
				self.update_interval(True)
		elif self.configuration.type == "report":
			if edge and edge == "trigger":
				print(f"Regenerating report {self.configuration.name}")
				core.schedule_update_hook(self.configuration.name)
		elif self.configuration.type == "hook":
			if edge and edge == "trigger":
				print(f"Scheduling hook {self.configuration.name}")
				if self.can_be_scheduled():
					self.schedule_asap = True
				
			
			
							


class MonitoringCore:
	def __init__(self, service_config_map):
		self.service_map = {}
		self.on_update_queue = []
		for name,config in service_config_map.items():
			self.service_map[name] = CheckInformation(config)
		self.setup()

	def setup(self):
		# Map out dependencies
		for service in self.service_map.values():
			for requirement in service.requires:
				if requirement in self.service_map:
					rs = self.service_map[requirement]
					if service.configuration.name not in rs.is_required_by:
						rs.is_required_by.append(service.configuration.name)
			for required_by in service.is_required_by:
				if required_by in self.service_map:
					rs = self.service_map[required_by]
					if service.configuration.name not in rs.requires:
						rs.requires.append(service.configuration.name)
		# Enable services without dependencies
		print("Initalizing unmet requirements arrays ...")
		counter = 0
		for service in self.service_map.values():
			service.unmet_requirements = copy.copy(service.requires)
			if service.configuration.type == "trigger":
				self.schedule_update_hook(service.configuration.name)
			elif service.configuration.type == "report":
				self.schedule_update_hook(service.configuration.name)
			print(f"* {service.configuration.name}: {service.unmet_requirements}")
			if service.is_active:
				counter += 1
		print(f"Enabled {counter} checks.")
		if counter == 0:
			print("No checks enabled, you have a faulty configuration")
		self.process_on_update_queue()

	def process_result(self, result):
		if result.name not in self.service_map:
			return
		is_ok = result.is_ok()
		print(f"Got result from {result.name} ok: {is_ok}")
		service = self.service_map[result.name]
		# Adjust interval
		service_result_changed = False
		if service.last_result == None:
			service_result_changed = True
		elif service.last_result.is_ok() != result.is_ok():
			service_result_changed = True
		service.update_interval(service_result_changed)
		# Update service state
		service.last_result = result
		service.is_scheduled = False
		# Update dependencies
		if service_result_changed:
			self.update_service(service)


	def schedule_update_hook(self, name):
		print(f"\x1b[37mScheduling service other\x1b[00m")
		if not name in self.on_update_queue:
			self.on_update_queue.append(name)

	def update_service(self, service):
		print(f"\x1b[01mUpdating service {service.configuration.name}\x1b[00m")
		is_initial_trigger = service.was_ok_on_last_update == None
		trigger_edge = get_edge_type(service.was_ok_on_last_update, service.is_ok())
		service.on_update(self)
		name = service.configuration.name
		for required_by in service.is_required_by:
			if required_by in self.service_map:
				rs = self.service_map[required_by]
				print(f"* {required_by}: {rs.unmet_requirements}")
				if trigger_edge:
					rs.process_trigger(trigger_edge, is_initial_trigger, self)
				update_hook = is_initial_trigger
				if service.is_ok():
					if name in rs.unmet_requirements:
						rs.unmet_requirements.remove(name)
						update_hook = True
				else:
					if name not in rs.unmet_requirements:
						rs.unmet_requirements.append(name)
						update_hook = True
				if update_hook:
					self.schedule_update_hook(rs.configuration.name)
				print(f"Updated unmet requirements of {rs.configuration.name} to {rs.unmet_requirements}")
		

	def process_on_update_queue(self):
		while self.on_update_queue:
			service_name = self.on_update_queue.pop()
			if service_name in self.service_map:
				service = self.service_map[service_name]
				self.update_service(service)

	# schedule all checks that need to be scheduled
	def schedule_due_checks(self, command_queue, response_queue):
		for service in self.service_map.values():
			if service.should_be_scheduled():
				#print(f"Scheduling {service.configuration.name} ...")
				service.mark_scheduled()
				command_queue.put(MonitorWrapper(service.configuration.name, service.configuration.command, response_queue))

	def get_next_schedule_datetime(self):
		now = datetime.now()
		next_schedule = now + timedelta(seconds=600)
		for service in self.service_map.values():
			if service.can_be_scheduled():
				if service.schedule_asap:
					return now
				t = now + service.interval
				if t < next_schedule:
					next_schedule = t
		return next_schedule

	def monitoring_loop(self, command_queue):
		response_queue = queue.Queue()
		while True:
			self.schedule_due_checks(command_queue, response_queue)
			command_queue.join()
			while not response_queue.empty():
				item = response_queue.get()
				self.process_result(item)
			self.process_on_update_queue()
			t = self.get_next_schedule_datetime()
			sleep_for = (t - datetime.now()).total_seconds()
			if sleep_for > 0:
				print(f"Will sleep for {sleep_for}s ...")
				time.sleep(sleep_for)
			
		
class Worker(threading.Thread):
	def __init__(self, command_queue, autostart=True, daemon=False):
		threading.Thread.__init__(self,daemon=daemon)
		self.running = True
		self.command_queue = command_queue
		if autostart:
			self.start()
		
	def run(self):
		while self.running:
			job = self.command_queue.get()
			if job is None:
				break
			#print(f"Running check {job.name}")
			job.do_check()
			self.command_queue.task_done()

command_queue = queue.Queue()

def read_configuration(file_name):
	with open(file_name, 'r') as file:
		data = json.load(file)
	service_map = {}
	for name, dictionary in data.items():
		service_map[name] = read_json_to_check_configuration(name, dictionary)
	return service_map

parser = argparse.ArgumentParser(
	prog = "Minimon",
	description = "A small monitoring service for managing local state"
)
parser.add_argument('configuration', nargs='+')
args = parser.parse_args()

path = args.configuration.pop()
print(f"Reading configuration file {path} ...")
config = read_configuration(path)
while args.configuration:
	path = args.configuration.pop()
	print(f"Reading configuration file {path} ...")
	overlay = read_configuration(path)
	for key,value in overlay.items():
		config[key] = value

core = MonitoringCore(config)

Worker(command_queue, daemon=True)

core.monitoring_loop(command_queue)
