#!/bin/bash

set -e

export LANG=C

[ ! -f "debian/control" ] && echo "No debian control file" && exit 2
[ ! -f "debian/changelog" ] && echo "No debian changelog file" && exit 2
[ ! -d ".git" ] && echo "Not a git repository" && exit 2


MISSING_DEPENDENCIES=""

check_command() {
	if ! command -v "$1" >/dev/null 2>/dev/null ; then
		MISSING_DEPENDENCIES="y"
		echo "Missing command: $1 (package: $2)"
	fi
}

check_command git git
check_command gbp git-buildpackage
check_command dh_clean debhelper

if [ -n "$MISSING_DEPENDENCIES" ] ; then
	echo "ERROR: Some dependencies are missing (see above), aborting" >&2
	exit 2
fi
unset MISSING_DEPENDENCIES

if ! git diff --quiet ; then
	echo "ERROR: Git repository has changes that are not comitted, debhelper doesn't like this." >&2
	exit 2
fi


HAS_UPSTREAM_BRANCH=""
PACKAGE_SORCE_NAME="$(grep "^Source: " debian/control | awk '{ print $2 }')"
PACKAGE_VERSION="$(head debian/changelog -n 1 | cut -d'(' -f2 | cut -d')' -f1)"
PACKAGE_VERSION_NO_PATCH="$(printf "%s" "$PACKAGE_VERSION" | sed 's/\-[^-]*$//' )"
#PACKAGE_PATCH_VERSION="$( printf "%s" "$PACKAGE_VERSION" | grep -Po '([^-]*)$' )"
PACKAGE_SOURCE_DIRNAME="$(basename "$(pwd)")"
CURRENT_GIT_BRANCH="$(git branch --show-current)"
ORIG_TAR_GZ_FILE_NAME="${PACKAGE_SORCE_NAME}_${PACKAGE_VERSION_NO_PATCH}.orig.tar.gz"

if git branch | grep -q upstream ; then
	HAS_UPSTREAM_BRANCH="y"
fi

[ -z "$PACKAGE_SORCE_NAME" ] && PACKAGE_SORCE_NAME="$PACKAGE_SOURCE_DIRNAME"

printf "Package Name: %s\n" "$PACKAGE_SORCE_NAME"
printf "Package Version: %s\n" "$PACKAGE_VERSION"
printf "Package Version (without patch): %s\n" "$PACKAGE_VERSION_NO_PATCH"
printf "Package Sourcedir Name: %s\n" "$PACKAGE_SOURCE_DIRNAME"
printf "Tarball Name: %s\n" "$ORIG_TAR_GZ_FILE_NAME"
printf "Current Branch: %s\n" "$CURRENT_GIT_BRANCH"

if [ -f "../$ORIG_TAR_GZ_FILE_NAME" ] ; then
	echo "Tarball »$ORIG_TAR_GZ_FILE_NAME« already present, reusing it."
elif [ -f "output_debian/$ORIG_TAR_GZ_FILE_NAME" ] ; then
	echo "Reusing tarball from output_debian."
	mv "output_debian/$ORIG_TAR_GZ_FILE_NAME" ../
else
	echo "No tarball found, ignoring."
fi

#clean up previous attempt
[ -d output_debian ] && rm -rf output_debian

if [ -n "$HAS_UPSTREAM_BRANCH" ] ; then
	UPSTREAM_BRANCH_OPTION="--git-upstream-branch=upstream"
else
	UPSTREAM_BRANCH_OPTION="--git-upstream-tree=SLOPPY"
fi

gbp buildpackage --git-cleaner="debuild --no-tgz-check -- clean" "$UPSTREAM_BRANCH_OPTION" --git-debian-branch="$CURRENT_GIT_BRANCH" -us -uc -sa
debuild --no-tgz-check -- clean

mkdir -p output_debian

# Move 
mv "../${PACKAGE_SORCE_NAME}_${PACKAGE_VERSION_NO_PATCH}"* output_debian/ || true

grep "^Package: " debian/control | awk '{ print $2 }' | xargs -I {} find .. -type f -name "{}*_$PACKAGE_VERSION_NO_PATCH*" | xargs -I {} mv "{}" output_debian/ || true

# make sure output_debian is in gitignore

[ -f .gitignore ] || touch .gitignore
if ! grep -q output_debian .gitignore ; then
	echo output_debian >> .gitignore
fi
