#!/bin/bash

set -e

REPREPRO_OPTIONS=""
DISTRIBUTIONS=""
CURRENT_RELEASE="$(lsb_release -c -s 2>/dev/null)"
IGNORE_WRONG_DISTRIBUTIONS=""

show_help() {
cat <<EOF
Usage: upload_deb.sh <<options>>

Run in a package build directory to upload the contents of the output_debian/ folder.

For internal use for the Treuchtlingen city administration only.

Options:
	--extra
		Upload to the extra repository

	-d, --distribution <distro>
		Upoad for a given release of a distribution only.
		Default: 'bullseye bookworm trixie'

	--ignore-wrong-distribution
		Allow uploading to a repository not listed in the changes file

EOF
}

while [[ "$#" -gt 0 ]]; do
	case "$1" in
		--extra)
			REPREPRO_OPTIONS="-b htdocs/extra/"
			shift 2;;
		-d|--distribution)
			DISTRIBUTIONS="$2"
			shift 2;;
		--ignore-wrong-distribution)
			IGNORE_WRONG_DISTRIBUTIONS="--ignore=wrongdistribution"
			shift 1;;
		--help) show_help; exit 0;;
		*) printf "Unknown option: %s\n" "$1"; exit 1;;
	esac
done

if [ ! -d output_debian ] ; then
	echo "No output_debian folder found"
	exit 1
fi

RELEASE_CHANNELS_FROM_CHANGELOG=" $(grep -oP -m 1 '.* [(][^(]+[)] \K[^;]+' debian/changelog ) "
if [ -z "$RELEASE_CHANNELS_FROM_CHANGELOG" ] ; then
	echo "Couldn't read release channels (distribution(s)) from changelog!"
	exit 1
fi

if grep 'Architecture: *\w*' debian/control | awk '$2 != "all" { exit 1 }' ; then
	if [ -z "$DISTRIBUTIONS" ] ; then
		if [ "$RELEASE_CHANNELS_FROM_CHANGELOG" != "unstable" ] ; then
			DISTRIBUTIONS="$RELEASE_CHANNELS_FROM_CHANGELOG"
		else
			DISTRIBUTIONS="bullseye bookworm trixie"
			echo "Architecture independed package, allowing upload to 'wrong' distribution."
			IGNORE_WRONG_DISTRIBUTIONS="--ignore=wrongdistribution"
		fi
	fi
else
	if [ -z "$DISTRIBUTIONS" ] ; then
		echo "Architecture dependent package, only uploading to single distribution release: $CURRENT_RELEASE"
		DISTRIBUTIONS="$CURRENT_RELEASE"
	fi
fi

matches() {
	TEXT="$1"
	shift 1
	printf "%s\n" "$TEXT" | grep -q "$@"
}
			
if ! matches "$DISTRIBUTIONS" '^[a-z ]*$' ; then
	echo "Please use only lowercase latin letters and spaces for the --distribution option."
	exit 1
fi

HAD_ERROR=""
if [ "$RELEASE_CHANNELS_FROM_CHANGELOG" = "unstable" ] ; then
	echo "Release channel from changelog is unstable, allowing upload to everything because of legacy reasons."
elif [ -n "$IGNORE_WRONG_DISTRIBUTIONS" ] ; then
	echo "Ignoring wrong distributions, not checking for changelog match."
else
	echo "Checking if release targets match changelog ..."
	for release in $DISTRIBUTIONS ; do
		if printf "%s" "$RELEASE_CHANNELS_FROM_CHANGELOG" | grep -qF " $release " ; then
			echo " * OK: $release is in latest changelog entry"
		else
			echo " * ERROR: $release is not in latest changelog entry"
			HAD_ERROR="y"
		fi
	done
fi

if [ -n "$HAD_ERROR" ] ; then
	echo "At least one target didn't match what the last changelog entry!"
	exit 1
fi

LOGIN_OF_PACKAGE_SERVER="packages@packages.treuchtlingen.stadt"

printf "Uploading Packages …\n"
rsync -aP output_debian/* "$LOGIN_OF_PACKAGE_SERVER":htdocs/_incoming/


printf "Running Reprepro …\n"
ssh "$LOGIN_OF_PACKAGE_SERVER" 'for DISTRI in '"$DISTRIBUTIONS"' ; do echo "Integrating for $DISTRI …"; reprepro '"$REPREPRO_OPTIONS"' -V '"$IGNORE_WRONG_DISTRIBUTIONS"' include $DISTRI htdocs/_incoming/*changes; echo; done'

printf "Moving Packages to done …\n"
ssh "$LOGIN_OF_PACKAGE_SERVER" 'mv htdocs/_incoming/*_* htdocs/_incoming/done/'
