#!/usr/bin/lua

local signature_template = [[
Mit freundlichen Grüßen

{ganzer_name}

-- 
{?position?}{position}, {?/?}{?abteilung?}{abteilung}{?/?}
{organisation_all_caps}, {strasse}, {plz} {ort}
Tel.: {telefon}
{website}
]]

local function read_file(name)
	local f = io.open(name)
	if not f then return nil end
	local content = f:read("a")
	f:close()
	return content
end

local function write_file(name, text)
	local f = io.open(name, "w+")
	if not f then return false end
	f:write(text)
	f:close()
	return true
end

local function parse_keys(text)
	local data = {}
	for line in text:gmatch("[^\n]+") do
		local key, value = line:match("^(.-)%s+(.*)$")
		if key and value then
			data[key] = value
		end
	end
	return data
end

local function fill_out_template(template, data, default)
	local default = default or "N/A"
	local template = template:gsub("{%?(.-)%?}(.-){%?/%?}", function(key, content)
		if data[key] and data[key] ~= "" then
			return content
		else
			return ""
		end
	end)
	local text = template:gsub("{(.-)}", function(key)
		return data[key] or default
	end)
	return text
end

local function parse_cli_args(args)
	local parsed_args = {}	
	local current_arg = parse_cli_args
	for _,arg in ipairs(args)	do
		if arg:match("^%-%-") then
			local key = arg:match("^..(.+)$")
			if not parsed_args[key] then
				parsed_args[key] = {}
			end
			current_arg = parsed_args[key]
		else
			current_arg[#current_arg+1] = arg
		end
	end
	return parsed_args
end

local function message(msg)
	io.stderr:write(tostring(msg).."\n")
end

local function get_pattern(pattern, key)
	local p = pattern:gsub("{{{key}}}", key)
	return p
end

local function read_keys(text, keylist, pattern)
	local out = {}
	for outname,pattname in pairs(keylist) do
		p = get_pattern(pattern, pattname)
		out[outname] = js_unescape(text:match(p))
	end
	return out
end

local function write_keys(text, data, pattern)
	for key, val in pairs(data) do
		local val = js_escape(val:gsub("%%",""))
		local p = get_pattern(pattern, key)
		val = p:gsub("%(%.%-%)",val)
		val = val:gsub("%%","")
		text = text:gsub(p, val)
	end
	return text
end

local args = parse_cli_args({...})

local signature_file = "/dev/stdout"
local data_file_path = "/dev/stdin"

if args["data-file"] then
	if #args["data-file"] >= 1 then
		data_file_path = args["data-file"][1]
	end
end

if args["to-signature-file"] then
	if #args["to-signature-file"] >= 1 then
		signature_file = args["to-signature-file"][1]
	end
end

local data = false

local text = read_file(data_file_path)
if text then
	print("Importiere Daten aus Datei …")
	data = parse_keys(text)
end

if args["help"] then
	message [[Usage: treuchtlingen-personal-data-mail-signature <options>
	--to-signature-file [<path/to/signature.txt>]
	--data-file [<path/to/datafile>]
	]]
	os.exit(2)
end

if data.organisation then
	data.organisation_all_caps = data.organisation:upper()
end

local signature_text = fill_out_template(signature_template, data)

write_file(signature_file, signature_text)
