#!/usr/bin/lua5.4

local function read_file(name)
	local f = io.open(name)
	if not f then return nil end
	local content = f:read("a")
	f:close()
	return content
end

local function write_file(name, text)
	local f = io.open(name, "w+")
	if not f then return false end
	f:write(text)
	f:close()
	return true
end

local function parse_keys(text)
	local data = {}
	for line in text:gmatch("[^\n]+") do
		local key, value = line:match("^(.-)%s+(.*)$")
		if key and value then
			data[key] = value
		end
	end
	return data
end

local function serialize_keys(data)
	local text = ""
	for key,value in pairs(data) do
		assert(not key:match("[%s\n]"), "Keys may not contain spaces or newlines")
		assert(not value:match("\n"), "Values may not contain newlines")
		text = text..key.." "..value.."\n"
	end
	return text
end

local function parse_cli_args(args)
	local parsed_args = {}	
	local current_arg = parse_cli_args
	for _,arg in ipairs(args)	do
		if arg:match("^%-%-") then
			local key = arg:match("^..(.+)$")
			if not parsed_args[key] then
				parsed_args[key] = {}
			end
			current_arg = parsed_args[key]
		else
			current_arg[#current_arg+1] = arg
		end
	end
	return parsed_args
end

local function message(msg)
	io.stderr:write(tostring(msg).."\n")
end


local outputtemplate = [[Persönliche Daten:
{ganzer_name} ({initialien})
{position}, {abteilung} bei {organisation}

Kontaktdaten:
E-Mail: {email}
Tel: {telefon} (Durchwahl: {tel_durchwahl})
Fax: {fax} (Durchwahl: {fax_durchwahl})
Zimmernummer: {zimmer}

Anschrift:
{organisation}
{ganzer_name}
{strasse}
{plz} {ort}
{land}, {region}

Generierte E-Mail Signatur (ungefähre Vorschau):
Mit freundlichen Grüßen

{ganzer_name}

-- 
{position}, {abteilung}
{organisation}, {strasse}, {plz} {ort}
Tel.: {telefon}
{website}
]]

local abteilungen = {
	"Zentrale Verwaltung",
	"Finanzverwaltung",
	"Bürgerbüro",
	"Stadtbauamt",
	"Bauhof",
	"Abwasseranlagen"
}

local websites = {
	"https://www.treuchtlingen.de",
	"https://www.tourismus-treuchtlingen.de",
}

local function fill_out_template(template, data, default)
	local default = default or "N/A"
	local text = template:gsub("{(.-)}", function(key)
		return data[key] or default
	end)
	return text
end

local function update_full_name(data)
	data.ganzer_name = data.titel or ""
	if data.vorname and data.vorname ~= "" then
		if data.ganzer_name == "" then
			data.ganzer_name = data.vorname
		else
			data.ganzer_name = data.ganzer_name.." "..data.vorname
		end
	end
	if data.nachname then
		if data.ganzer_name == "" then
			data.ganzer_name = data.nachname
		else
			data.ganzer_name = data.ganzer_name.." "..data.nachname
		end
	end
end

local function ask(question, default, suggestions)
	if suggestions and #suggestions > 0 then
		for n,s in ipairs(suggestions) do
			print("[+"..n.."] "..tostring(s))
		end
	end
	if default and default ~= "" then
		io.write(question.." ("..default.."): ")	
	else
		io.write(question..": ")
	end
	local answer = io.read()
	-- parse suggestion
	if suggestions and #suggestions > 0 then
		if answer:match("^%+[0-9]+$") then
			return suggestions[tonumber(answer:match("^%+([0-9]+)$")) or false] or answer
		end
	end
	if answer:match("^%s*$") then
		return default
	elseif answer:match("^%s*%-%s*$") then
		return ""
	else
		return answer:match("^%s*(.-)%s*$")
	end
end

local function ask_yes_no(question, default)
	local prompt = " [j/N]: "
	if default then prompt = " [J/n]: " end
	io.write(question..prompt)
	local answer = io.read()
	if answer:lower():match("^%s*$") then
		return default
	else
		return not not answer:match("^[JjYy]")
	end
end

local function ui(data)
	print("Durch die Tastenkombination Strg+c oder durch Schließen des Konsolenfensters wird das Programm OHNE zu speichern sofort beendet.")
	while true do
		print("-----")
		update_full_name(data)
		print(fill_out_template(outputtemplate, data))
		if not ask_yes_no("Sollen diese Angaben geändert werden?", true) then
			return
		end
		print("---")
		print("[1] Name und Titel")
		print("[2] Organisation und Tätigkeit")
		print("[3] Kontakdaten")
		print("[4] Anschrift")
		print("[*] Alles")
		local op = ask("Was wollen sie anpassen? [1/2/3/4/*]")
		print("[-] eingeben um ein Feld zu leeren.")
		if op == "1" or op == "*" then
			data.vorname = ask("Vorname", data.vorname)
			data.nachname = ask("Nachname", data.nachname)
			data.titel = ask("Titel", data.titel)
			data.initialien = ask("Namenskürzel", data.initialien)
			update_full_name(data)
		end
		if op == "2" or op == "*" then
			data.organisation = ask("Organisation", data.organisation, {"Stadt Treuchtlingen", "Kur- und Touristinformation Stadt Treuchtlingen"})
			print("Für Abteilung und Position bitte an Telefonliste orientieren.")
			data.abteilung = ask("Abteilung", data.abteilung, abteilungen)
			data.position = ask("Tätigkeit", data.position)
			print("Bitte Relevante Website im format https://… angeben. (Kein '/' am Ende!)")
			data.website = ask("Website", data.website, websites)
		end
		if op == "3" or op == "*" then
			local suggested_email = data.email
			if data.email then
				if data.email:match("@.+") then
					 suggested_email = data.vorname:lower().."."..data.nachname:gsub("ä","ae"):gsub("ü","ue"):gsub("ö","oe"):gsub("ß","ss"):lower()..data.email:match("@.+")
				end
			end
			if suggested_email ~= data.email then
				print("[+] eingeben um folgenden Vorschlag anzunehmen:")
				print("  "..suggested_email)
			end
			data.email = ask("E-Mail", data.email)
			if data.email == "+" then data.email = suggested_email end
			print("Telefonnummern bitte im Format: vorwahl-nummer-durchwahl angeben")
			data.telefon = ask("Telefon", data.telefon)
			if data.telefon == "" then data.telefon = nil end
			if data.telefon then
				data.tel_durchwahl = (data.telefon and data.telefon:match("%d+%-%d+%-?(%d*)")) or nil
				if data.tel_durchwahl == "" then data.tel_durchwahl = nil end
			else
				data.tel_durchwahl = nil
			end
			local suggested_fax_durchwahl = (data.tel_durchwahl and "9"..data.tel_durchwahl) or nil
			if suggested_fax_durchwahl and suggested_fax_durchwahl ~= data.fax_durchwahl then
				print("[+] eingeben um folgenden Vorschlag anzunehmen:")
				print("  "..suggested_fax_durchwahl)
			end
			data.fax_durchwahl = ask("Faxdurchwahl", data.fax_durchwahl)
			if data.fax_durchwahl == "" then data.fax_durchwahl = nil end
			if data.fax_durchwahl == "+" then data.fax_durchwahl = suggested_fax_durchwahl end
			if data.telefon and data.tel_durchwahl and data.fax_durchwahl and data.fax_durchwahl ~= "" and data.fax_durchwahl then
				data.fax = data.telefon:sub(1, data.telefon:len() - data.tel_durchwahl:len())..data.fax_durchwahl
			else
				data.fax = nil
			end
			data.zimmer = ask("Zimmernummer", data.zimmer)
		end
		if op == "4" or op == "*" then
			print("Anschrift:")
			data.strasse = ask("Straße:", data.strasse, {"Hauptstraße 31", "Heinrich-Aurnhammer-Str. 3", "Dietfurter Straße 4", "Kanalstraße 51"})
			data.plz = ask("Postleitzahl:", data.plz)
			data.ort = ask("Ort:", data.ort)
			data.land = ask("Land:", data.land)
			data.region = ask("Region:", data.region)
		end
	end
end

local args = parse_cli_args({...})

local data_file_path = os.getenv("HOME").."/.config/treuchtlingen-personal-data"
local data_import_fallback_command = false

local savehooks = args.savehooks or {}

if args["data-file"] then
	if #args["data-file"] >= 1 then
		data_file_path = args["data-file"][1]
	end
end

if args["import-fallback-command"] then
	if #args["import-fallback-command"] >= 1 then
		data_import_fallback_command = args["import-fallback-command"][1]
	end
end

local data = false

local text = read_file(data_file_path)
if text then
	print("Importiere Daten aus Datei …")
	data = parse_keys(text)
elseif data_import_fallback_command then
	print("Importiere Daten aus Fallbackbefehl …")
	local f = io.popen(data_import_fallback_command)
	if f then
		local text = f:read("a")
		if text then
			data = parse_keys(text)
		end
	end
end

if data then
	print("Import erfolgreich.")
else
	print("Datenimport fehlgeschlagen!")
	if ask_yes_no("Mit leeren Datensatz fortsetzen?", true) then
		data = {}
	else
		exit(130)
	end
end

ui(data)

if ask_yes_no("Soll diese Konfiguration gespeichert werden?", true) then
	while true do
		print("Schreibe Konfiguration in "..data_file_path)
		if write_file(data_file_path, serialize_keys(data)) then
			print("Geschrieben!")
			break
		else
			print("Leider ist die Konfiguration nicht schreibbar.")
			if not ask_yes_no("Noch einmal versuchen?", true) then
				os.exit(2)
			end
		end
	end
	
	for _,hook_command in ipairs(savehooks or {}) do
		print("Benachrichtige: "..hook_command)
		if not os.execute(hook_command) then
			print("Benachrichtigung fehlgeschlagen, Änderungen wurden möglicherweise nicht vollständig übernommen.")
		end
	end
	print("Fertig!")
end
