#!/usr/bin/lua5.4

local function read_file(name)
	local f = io.open(name)
	if not f then return nil end
	local content = f:read("a")
	f:close()
	return content
end

local function write_file(name, text)
	local f = io.open(name, "w+")
	if not f then return false end
	f:write(text)
	f:close()
	return true
end

local function parse_keys(text)
	local data = {}
	for line in text:gmatch("[^\n]+") do
		local key, value = line:match("^(.-)%s+(.*)$")
		if key and value then
			data[key] = value
		end
	end
	return data
end

local function serialize_keys(data)
	local text = ""
	for key,value in pairs(data) do
		assert(not key:match("[%s\n]"), "Keys may not contain spaces or newlines")
		assert(not value:match("\n"), "Values may not contain newlines")
		text = text..key.." "..value.."\n"
	end
	return text
end

local function parse_cli_args(args)
	local parsed_args = {}	
	local current_arg = parse_cli_args
	for _,arg in ipairs(args)	do
		if arg:match("^%-%-") then
			local key = arg:match("^..(.+)$")
			if not parsed_args[key] then
				parsed_args[key] = {}
			end
			current_arg = parsed_args[key]
		else
			current_arg[#current_arg+1] = arg
		end
	end
	return parsed_args
end

local function message(msg)
	io.stderr:write(tostring(msg).."\n")
end


local pattern_libreoffice = '<item oor:path="/org%.openoffice%.UserProfile/Data"><prop oor:name="{{{key}}}" oor:op="fuse"><value>(.-)</value></prop></item>'

local keys_libreoffice = {
	vorname = "givenname",
	--telefon_privat = "homephone",
	initialien = "initials",
	ort = "l",
	email = "mail",
	organisation = "o",
	position = "position",
	plz = "postalcode",
	nachname = "sn",
	land = "st",
	strasse = "street",
	tel_durchwahl = "telephonenumber",
	zimmer = "title", --eigentlich: titel
	fax_durchwahl = "facsimiletelephonenumber",
	region = "c",
}

local function xml_escape(text)
	return text:gsub("[&<>]", {
		["&"] = "&amp;",
		["<"] = "&lt;",
		[">"] = "&gt;",
	})
end

local function xml_unescape(text)
	if not text then return nil end
	return text:gsub("&...?;", {
		["&amp;"] = "&",
		["&lt;"] = "<",
		["&gt;"] = ">",
	})
end

local function get_pattern(pattern, key)
	local p = pattern:gsub("{{{key}}}", key)
	return p
end

local function read_keys(text, keylist, pattern)
	local out = {}
	for outname,pattname in pairs(keylist) do
		p = get_pattern(pattern, pattname)
		out[outname] = xml_unescape(text:match(p))
	end
	return out
end

local function write_keys(text, keylist, pattern, data)
	for name, val in pairs(data) do
		local key = keylist[name]
		if key then
			local val = xml_escape(val:gsub("%%",""))
			local p = get_pattern(pattern, key)
			val = p:gsub("%(%.%-%)",val)
			val = val:gsub("%%","")
			text = text:gsub(p, val)
		end
	end
	return text
end

local function fill_out_template(template, data, default)
	local default = default or "N/A"
	local text = template:gsub("{(.-)}", function(key)
		return data[key] or default
	end)
	return text
end


local args = parse_cli_args({...})

local config_path = os.getenv("HOME").."/.config/libreoffice/4/user/registrymodifications.xcu"
local operation = "from-libreoffice"
local data_file_path = "/dev/stdout"

for _,op in ipairs({"to-libreoffice", "from-libreoffice"}) do
	if args[op] then
		operation = op
		if #args[op] >= 1 then
			config_path = args[op][1]
		end
		if op == "to-libreoffice" then
			data_file_path = "/dev/stdin"
		end
	end
end

if args["data-file"] then
	if #args["data-file"] >= 1 then
		data_file_path = args["data-file"][1]
	end
end

if operation == "from-libreoffice" then
	local config_text = read_file(config_path)
	if not config_text then
		message("Unable to read libreoffice configuration!")
		os.exit(1)
	end
	local data = read_keys(config_text, keys_libreoffice, pattern_libreoffice)
	local text = serialize_keys(data)
	if not write_file(data_file_path, text) then
		message("Unable to write to data file!")
		os.exit(2)
	end
elseif operation == "to-libreoffice" then
	local text = read_file(data_file_path)
	if not text then
		message("Unable to read data file!")
		os.exit(1)
	end
	local data = parse_keys(text)
	local config_text = read_file(config_path)
	if not config_text then
		message("Unable to read libreoffice configuration!")
		os.exit(1)
	end
	if not write_file(config_path, write_keys(config_text, keys_libreoffice, pattern_libreoffice, data)) then
		message("Unable to write to libreoffice configuration")
		os.exit(2)
	end
else
	message [[Usage: treuchtlingen-personal-data-libreoffice <options>
	--to-libreoffice [<path/to/registrymodifications.xcu>]
	--from-libreoffice [<path/to/registrymodifications.xcu>]
	--data-file [<path/to/datafile>]
	]]
end
