#!/usr/bin/lua

local function read_file(name)
	local f = io.open(name)
	if not f then return nil end
	local content = f:read("a")
	f:close()
	return content
end

local function write_file(name, text)
	local f = io.open(name, "w+")
	if not f then return false end
	f:write(text)
	f:close()
	return true
end

local function parse_keys(text)
	local data = {}
	for line in text:gmatch("[^\n]+") do
		local key, value = line:match("^(.-)%s+(.*)$")
		if key and value then
			data[key] = value
		end
	end
	return data
end

local function serialize_keys(data)
	local text = ""
	for key,value in pairs(data) do
		assert(not key:match("[%s\n]"), "Keys may not contain spaces or newlines")
		assert(not value:match("\n"), "Values may not contain newlines")
		text = text..key.." "..value.."\n"
	end
	return text
end

local function parse_cli_args(args)
	local parsed_args = {}	
	local current_arg = parse_cli_args
	for _,arg in ipairs(args)	do
		if arg:match("^%-%-") then
			local key = arg:match("^..(.+)$")
			if not parsed_args[key] then
				parsed_args[key] = {}
			end
			current_arg = parsed_args[key]
		else
			current_arg[#current_arg+1] = arg
		end
	end
	return parsed_args
end

local function message(msg)
	io.stderr:write(tostring(msg).."\n")
end

local function uri_escape(text)
	local escaped_text = text:gsub("[^A-Za-z0-9%-%./]", function (char)
		return "%"..("%x"):format(string.byte(char))
	end)
	return escaped_text
end

local pattern_thunderbird = 'user_pref%("{{{key}}}", "(.-)"%);'

local function get_thunderbird_prefs_data(data)
	local mailname = data.email:match("^(.-)@")
	local escaped_mail = uri_escape(data.email)
	local home = os.getenv("HOME"):gsub("/$","")
	local user = os.getenv("USER")
	local imap_uri = "imap://"..escaped_mail.."@imap.treuchtlingen.stadt"
	local profile_dir = home.."/.thunderbird/rathaus.default"
	return {
		["mail.identity.id1.fullName"] = data.ganzer_name,
		["mail.server.server1.name"] = data.email,
		["mail.identity.id1.useremail"] = data.email,
		["mail.identity.id1.stationery_folder"] = imap_uri.."/Templates",
		["mail.identity.id1.archive_folder"] = imap_uri.."/Archives",
		["mail.identity.id1.fcc_folder"] = imap_uri.."/Sent",
		["mail.identity.id1.draft_folder"] = imap_uri.."/Drafts",
		["mail.identity.id1.sig_file"] = "/vorlagen/e-mail-signaturen/"..user..".txt",
		["mail.root.imap"] = profile_dir.."/ImapMail",
		["mail.server.server1.spamActionTargetAccount"] = imap_uri,
		["mail.server.server1.directory"] = profile_dir.."/ImapMail/imap.treuchtlingen.stadt",
		["mail.server.server2.directory"] = profile_dir.."/Mail/Local Folders",
		["mail.server.server3.directory"] = profile_dir.."/Mail/Feeds",
		["mail.server.server1.userName"] = email,
		["mail.server.server1.realuserName"] = user,
		["mail.smtpserver.smtp1.username"] = user,
	}
end

local function js_escape(text)
	return text:gsub("\\","\\\\"):gsub("[\"]", {
		["\""] = "\\\"",
		["\n"] = "\\\n",
		["\t"] = "\\\t",
	})
end

local function js_unescape(text)
	if not text then return nil end
	return text:gsub("&...?;", {
		["\\\""] = "\"",
		["\\\n"] = "\n",
		["\\\t"] = "\t",
	}):gsub("\\\\", "\\")
end

local function get_pattern(pattern, key)
	local p = pattern:gsub("{{{key}}}", key)
	return p
end

local function read_keys(text, keylist, pattern)
	local out = {}
	for outname,pattname in pairs(keylist) do
		p = get_pattern(pattern, pattname)
		out[outname] = js_unescape(text:match(p))
	end
	return out
end

local function write_keys(text, data, pattern)
	for key, val in pairs(data) do
		local val = js_escape(val:gsub("%%",""))
		local p = get_pattern(pattern, key)
		val = p:gsub("%(%.%-%)",val)
		val = val:gsub("%%","")
		text = text:gsub(p, val)
	end
	return text
end

local args = parse_cli_args({...})

local config_path = os.getenv("HOME").."/.thunderbird/rathaus.default/prefs.js"
local operation = "to-thunderbird"
local data_file_path = "/dev/stdin"

for _,op in ipairs({"to-thunderbird"}) do
	if args[op] then
		operation = op
		if #args[op] >= 1 then
			config_path = args[op][1]
		end
		if op == "to-thunderbird" then
			data_file_path = "/dev/stdin"
		end
	end
end

if args["data-file"] then
	if #args["data-file"] >= 1 then
		data_file_path = args["data-file"][1]
	end
end

if operation == "to-thunderbird" then
	local text = read_file(data_file_path)
	if not text then
		message("Unable to read data file!")
		os.exit(1)
	end
	local data = parse_keys(text)
	local config_text = read_file(config_path)
	if not config_text then
		message("Unable to read thunderbird configuration!")
		os.exit(1)
	end
	if not write_file(config_path, write_keys(config_text, get_thunderbird_prefs_data(data), pattern_thunderbird)) then
		message("Unable to write to thunderbird configuration")
		os.exit(2)
	end
else
	message [[Usage: treuchtlingen-personal-data-thunderbird <options>
	--to-thunderbird [<path/to/prefs.js>]
	--data-file [<path/to/datafile>]
	]]
end
