#include "invoiceoperations.h"
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QProcess>
#include <unistd.h>
#include <pwd.h>

InvoiceOperations::InvoiceOperations(QObject *parent)
    : QObject(parent)
{
}

bool InvoiceOperations::createInvoiceFolder(
    const QString &invoiceFile,
    const QString &targetBasePath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    if (invoiceFile.isEmpty() || targetBasePath.isEmpty() ||
        receiptStamp.isEmpty() || date.isEmpty() || sender.isEmpty() ||
        invoiceNumber.isEmpty() || budgetCode.isEmpty() ||
        user.isEmpty()) {
        workflowLogger.logError("Validation", "Required fields missing");
        return false;
    }

    QString folderName = buildFolderName(date, sender, invoiceNumber,
                                          budgetCode, user, reviewed, comment);
    QString targetPath = targetBasePath + "/" + folderName;

    workflowLogger.setLogPath(targetPath);

    QDir dir;
    if (!dir.mkpath(targetPath)) {
        workflowLogger.logError("Folder", "Failed to create: " + targetPath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFolderCreated(targetPath);

    QString cleanInvoiceFile = invoiceFile;
    if (cleanInvoiceFile.startsWith("file://")) {
        cleanInvoiceFile = cleanInvoiceFile.mid(7);
    }

    QFile file(cleanInvoiceFile);
    if (!file.exists()) {
        workflowLogger.logError("Source", "File not found: " + cleanInvoiceFile);
        workflowLogger.writeLog();
        return false;
    }

    QFileInfo fileInfo(cleanInvoiceFile);
    QString targetFilePath = targetPath + "/" + fileInfo.fileName();

    if (!file.rename(targetFilePath)) {
        workflowLogger.logError("Move", "Failed to move file to: " + targetFilePath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFileMoved(cleanInvoiceFile, targetFilePath);

    createDirectoryFile(targetPath, receiptStamp, date, sender, invoiceNumber, budgetCode,
                        user, reviewed, comment, amount, folderIcon);

    return true;
}

void InvoiceOperations::createDirectoryFile(
    const QString &folderPath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    QString directoryFilePath = folderPath + "/.directory";
    QFile directoryFile(directoryFilePath);

    if (!directoryFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return;
    }

    QTextStream out(&directoryFile);

    out << "[Desktop Entry]\n";
    out << "Icon=" << folderIcon << "\n";

    out << "\n[Rechnungslauf]\n";
    out << "ReceiptDate=" << receiptStamp << "\n";
    out << "IssueDate=" << date << "\n";
    out << "Seller=" << sender << "\n";
    out << "ID=" << invoiceNumber << "\n";
    out << "BuyerReference=" << budgetCode << "\n";
    out << "ProcessedBy=" << user << "\n";
    out << "Reviewed=" << (reviewed ? "yes" : "no") << "\n";

    if (!comment.isEmpty()) {
        out << "Note=" << comment << "\n";
    }

    if (!amount.isEmpty()) {
        out << "PayableAmount=" << amount << "\n";
    }

    directoryFile.close();
}

QString InvoiceOperations::getCurrentUser()
{
    QString user = qgetenv("USER");

    if (user.isEmpty()) {
        struct passwd *pw = getpwuid(getuid());
        if (pw != nullptr) {
            user = QString::fromUtf8(pw->pw_name);
        }
    }

    return user;
}

QString InvoiceOperations::buildFolderName(
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool /* reviewed */,
    const QString & /* comment */)
{
    QString cleanSender = QString(sender).replace(" ", "_");
    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "_");

    return QString("%1_%2_%3_%4_%5")
        .arg(date)
        .arg(cleanSender)
        .arg(cleanInvoiceNumber)
        .arg(budgetCode)
        .arg(user);
}

void InvoiceOperations::openFile(const QString &filePath)
{
    QProcess::startDetached("xdg-open", QStringList() << filePath);
}

void InvoiceOperations::logChecksum(const QString &filePath, const QString &checksum)
{
    workflowLogger.logChecksumCreated(filePath, checksum);
}

void InvoiceOperations::logStampedPdf(const QString &filePath)
{
    workflowLogger.logStampedPdfCreated(filePath);
}

void InvoiceOperations::logPinValidated(const QString &user)
{
    workflowLogger.logPinValidated(user);
}

void InvoiceOperations::logReviewedStatus(bool reviewed, const QString &user)
{
    workflowLogger.logReviewedStatus(reviewed, user);
}

void InvoiceOperations::logBudgetCode(const QString &budgetCode)
{
    workflowLogger.logBudgetCode(budgetCode);
}

void InvoiceOperations::logError(const QString &operation, const QString &message)
{
    workflowLogger.logError(operation, message);
}

void InvoiceOperations::finalizeLog()
{
    workflowLogger.writeLog();
}

bool InvoiceOperations::folderExists(const QString &targetBasePath, const QString &folderName)
{
    QString targetPath = targetBasePath + "/" + folderName;
    QDir dir(targetPath);
    return dir.exists();
}

bool InvoiceOperations::invoiceNumberExists(const QString &targetBasePath, const QString &invoiceNumber)
{
    if (invoiceNumber.isEmpty() || targetBasePath.isEmpty()) {
        return false;
    }

    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "_");
    QDir dir(targetBasePath);

    if (!dir.exists()) {
        return false;
    }

    QStringList entries = dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot);
    for (const QString &entry : entries) {
        QStringList parts = entry.split("_");
        if (parts.size() >= 5 && parts[2] == cleanInvoiceNumber) {
            return true;
        }
    }

    return false;
}

bool InvoiceOperations::createNoticeFile(const QString &folderPath, const QString &date, const QString &noticeText)
{
    if (noticeText.isEmpty()) {
        return true; // Nothing to create
    }

    QString noticeFilePath = folderPath + "/" + date + "_Notice.txt";
    QFile noticeFile(noticeFilePath);

    if (!noticeFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        workflowLogger.logError("Notice", "Failed to create notice file: " + noticeFilePath);
        return false;
    }

    QTextStream out(&noticeFile);
    out << noticeText << "\n";
    noticeFile.close();

    return true;
}
