import QtQuick
import QtQuick.Controls as Controls
import QtQuick.Layouts
import QtQuick.Dialogs
import QtCore
import org.kde.kirigami as Kirigami

Kirigami.ApplicationWindow {
    id: root

    title: i18n("XRDialog - Invoice Metadata")
    width: 950
    height: 790
    minimumWidth: 900
    minimumHeight: 760
    visible: true

    pageStack.globalToolBar.style: Kirigami.ApplicationHeaderStyle.ToolBar
    pageStack.globalToolBar.showNavigationButtons: Kirigami.ApplicationHeaderStyle.NoNavigationButtons

    property string selectedInvoice: initialInvoice || ""
    property bool allFieldsFilled: false
    property bool isBusy: false
    property string busyMessage: ""
    property bool isEditMode: false
    property string editFolderPath: ""

    function extractBudgetCode() {
        if (!budgetCodeField) {
            return ""
        }

        let text = budgetCodeField.text
        if (text.includes(",")) {
            return text.split(",")[0].trim()
        }
        return text
    }

    function buildFolderName() {
        if (!dateField || !invoiceNumField || !senderField ||
            !budgetCodeField || !userField) {
            return ""
        }

        let date = dateField.text || "YYYY-MM-DD"
        let sender = (senderField.text || "Seller").replace(/ /g, "_")
        let num = (invoiceNumField.text || "ID").replace(/ /g, "_")
        let budgetCode = extractBudgetCode() || "BudgetCode"
        let user = userField.text || "user"

        return date + "_" + sender + "_" + num + "_" + budgetCode + "_" + user
    }

    function isValidDate(dateStr) {
        // Check format YYYY-MM-DD
        let match = dateStr.match(/^(\d{4})-(\d{2})-(\d{2})$/)
        if (!match) return false
        let year = parseInt(match[1])
        let month = parseInt(match[2])
        let day = parseInt(match[3])
        if (year < 2000 || year > 2100) return false
        if (month < 1 || month > 12) return false
        if (day < 1 || day > 31) return false
        return true
    }

    function validateFields() {
        allFieldsFilled = (
            selectedInvoice !== "" &&
            isValidDate(receiptStampField.text) &&
            pinField.text !== "" &&
            isValidDate(dateField.text) &&
            senderField.text !== "" &&
            invoiceNumField.text !== "" &&
            extractBudgetCode() !== "" &&
            userField.text !== ""
        )
    }

    // Main page
    pageStack.initialPage: Kirigami.Page {
        id: mainPage
        title: i18n("Invoice Metadata")

        actions: [
            Kirigami.Action {
                text: i18n("Open")
                icon.name: "document-open"
                tooltip: i18n("Open Incoming Invoice")
                onTriggered: fileDialog.open()
            },
            Kirigami.Action {
                text: i18n("Show Incoming Invoice")
                icon.name: "view-preview"
                displayHint: Kirigami.DisplayHint.IconOnly
                tooltip: i18n("Show Incoming Invoice")
                enabled: root.selectedInvoice !== ""
                onTriggered: {
                    let cleanPath = root.selectedInvoice
                    if (cleanPath.startsWith("file://")) {
                        cleanPath = cleanPath.substring(7)
                    }
                    invoiceOps.openFile(cleanPath)
                }
            },
            Kirigami.Action {
                text: i18n("Preview Stamped Invoice")
                icon.name: "zoom-original"
                displayHint: Kirigami.DisplayHint.IconOnly
                tooltip: i18n("Preview Stamped Invoice")
                enabled: root.selectedInvoice !== "" && root.selectedInvoice.toLowerCase().endsWith(".pdf") && !root.isBusy
                onTriggered: {
                    if (!configMgr.validatePin(pinField.text)) {
                        root.showPassiveNotification(i18n("Error: Invalid PIN"))
                        return
                    }
                    root.isBusy = true
                    root.busyMessage = i18n("Creating preview...")
                    previewTimer.start()
                }
            },
            Kirigami.Action {
                text: i18n("Help")
                icon.name: "help-about"
                displayHint: Kirigami.DisplayHint.IconOnly
                tooltip: i18n("Help / Info")
                enabled: configMgr && configMgr.getHelpUrl() !== ""
                onTriggered: {
                    Qt.openUrlExternally(configMgr.getHelpUrl())
                }
            }
        ]

        Component.onCompleted: {
            if (configMgr && configMgr.isConfigured()) {
                if (!configMgr.checkDirectories()) {
                    directoryErrorMessage.text = configMgr.getConfigErrors()
                    directoryErrorMessage.visible = true
                }
            }
        }

        ColumnLayout {
            anchors.fill: parent
            anchors.margins: Kirigami.Units.largeSpacing
            anchors.bottomMargin: Kirigami.Units.smallSpacing
            spacing: Kirigami.Units.largeSpacing

            Kirigami.InlineMessage {
                id: configErrorMessage
                visible: configMgr ? !configMgr.isConfigured() : true
                Layout.fillWidth: true
                type: Kirigami.MessageType.Error
                text: {
                    if (!configMgr) {
                        return i18n("Configuration manager not initialized")
                    }
                    if (!configMgr.isConfigured()) {
                        let errors = configMgr.getConfigErrors()
                        if (errors !== "") {
                            return errors
                        }
                        return i18n("Configuration not found. Please create ~/.config/xrdialog/config file.")
                    }
                    return ""
                }
            }

            Kirigami.InlineMessage {
                id: directoryErrorMessage
                visible: false
                Layout.fillWidth: true
                type: Kirigami.MessageType.Error
                text: ""
            }

            Kirigami.InlineMessage {
                id: noFileMessage
                visible: root.selectedInvoice === "" && configMgr && configMgr.isConfigured()
                Layout.fillWidth: true
                type: Kirigami.MessageType.Information
                text: i18n("Please open an invoice file to begin.")
            }

            Kirigami.FormLayout {
                id: formLayout
                enabled: configMgr && configMgr.isConfigured() && root.selectedInvoice !== ""
                Layout.fillWidth: true
                wideMode: true

                Kirigami.Separator {
                    Kirigami.FormData.label: i18n("Invoice Data")
                    Kirigami.FormData.isSection: true
                }

                Controls.TextField {
                    id: invoiceFileField
                    Kirigami.FormData.label: i18n("Invoice File:")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    readOnly: true
                    placeholderText: i18n("Select invoice file...")
                    text: {
                        if (root.selectedInvoice.startsWith("file://")) {
                            return root.selectedInvoice.substring(7)
                        }
                        return root.selectedInvoice
                    }
                }

                Controls.TextField {
                    id: receiptStampField
                    Kirigami.FormData.label: i18n("Receipt Date:")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    placeholderText: i18n("YYYY-MM-DD")
                    text: Qt.formatDate(new Date(), "yyyy-MM-dd")
                    inputMask: "9999-99-99"
                    onTextChanged: root.validateFields()
                }

                Controls.TextField {
                    id: dateField
                    Kirigami.FormData.label: i18n("Issue Date:")
                    placeholderText: i18n("YYYY-MM-DD")
                    text: Qt.formatDate(new Date(), "yyyy-MM-dd")
                    inputMask: "9999-99-99"
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    onTextChanged: root.validateFields()
                }

                Controls.TextField {
                    id: senderField
                    Kirigami.FormData.label: i18n("Seller:")
                    placeholderText: i18n("Company/Person name")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    onTextChanged: root.validateFields()
                }

                Controls.TextField {
                    id: invoiceNumField
                    Kirigami.FormData.label: i18n("Invoice ID:")
                    placeholderText: i18n("e.g., 12345")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    onTextChanged: root.validateFields()
                }

                Item {
                    id: budgetCodeItem
                    Kirigami.FormData.label: i18n("Buyer Reference:")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    implicitHeight: budgetCodeField.implicitHeight

                    property alias editText: budgetCodeField.text
                    property alias popup: budgetCodePopup

                    Controls.TextField {
                        id: budgetCodeField
                        anchors.fill: parent
                        placeholderText: i18n("Type to search...")
                        onTextChanged: {
                            root.validateFields()
                            if (text.length > 0 && activeFocus) {
                                budgetCodeItem.popup.open()
                            } else {
                                budgetCodeItem.popup.close()
                            }
                        }
                        onActiveFocusChanged: {
                            if (!activeFocus) {
                                budgetCodeItem.popup.close()
                            }
                        }
                    }

                    Controls.Popup {
                        id: budgetCodePopup
                        y: budgetCodeField.height
                        width: budgetCodeField.width
                        height: Math.min(suggestionList.contentHeight + 10, 200)
                        padding: 5

                        ListView {
                            id: suggestionList
                            anchors.fill: parent
                            clip: true
                            model: {
                                if (!configMgr) return []
                                let all = configMgr.getBudgetCodes()
                                let filter = budgetCodeField.text.toLowerCase()
                                if (filter === "") return all
                                return all.filter(function(item) {
                                    return item.toLowerCase().includes(filter)
                                })
                            }
                            delegate: Controls.ItemDelegate {
                                width: suggestionList.width
                                text: modelData
                                highlighted: ListView.isCurrentItem
                                onClicked: {
                                    let code = modelData
                                    if (code.includes(",")) {
                                        code = code.split(",")[0].trim()
                                    }
                                    budgetCodeField.text = code
                                    budgetCodeItem.popup.close()
                                    root.validateFields()
                                }
                            }
                        }
                    }
                }

                RowLayout {
                    Kirigami.FormData.label: i18n("Payable Amount (optional):")
                    spacing: Kirigami.Units.smallSpacing
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40

                    Controls.TextField {
                        id: amountField
                        placeholderText: i18n("e.g., 1234.56")
                        Layout.fillWidth: true
                        Layout.preferredWidth: 2
                        validator: RegularExpressionValidator {
                            regularExpression: /^\d*([.,]\d{0,2})?$/
                        }
                    }

                    Controls.TextField {
                        id: currencyField
                        Layout.fillWidth: true
                        Layout.preferredWidth: 1
                        text: "EUR"
                        readOnly: true
                        enabled: false
                        horizontalAlignment: Text.AlignHCenter
                    }
                }

                // Processing Section !FIXME! Should become a Tab soon
                Kirigami.Separator {
                    Kirigami.FormData.label: i18n("Processing")
                    Kirigami.FormData.isSection: true
                }

                Controls.CheckBox {
                    id: reviewedCheckbox
                    Kirigami.FormData.label: i18n("Reviewed:")
                    text: i18n("Invoice has been reviewed")
                    checked: false
                    onCheckedChanged: root.validateFields()
                }

                Controls.TextField {
                    id: commentField
                    Kirigami.FormData.label: i18n("Note (optional):")
                    placeholderText: i18n("e.g., project-a")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    onTextChanged: root.validateFields()
                }

                RowLayout {
                    Kirigami.FormData.label: i18n("User / PIN:")
                    spacing: Kirigami.Units.smallSpacing
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40

                    Controls.TextField {
                        id: userField
                        Layout.fillWidth: true
                        Layout.preferredWidth: 2
                        placeholderText: i18n("Linux username")
                        onTextChanged: root.validateFields()

                        Component.onCompleted: {
                            if (invoiceOps) {
                                text = invoiceOps.getCurrentUser()
                            }
                        }
                    }

                    Controls.TextField {
                        id: pinField
                        Layout.fillWidth: true
                        Layout.preferredWidth: 1
                        placeholderText: i18n("PIN")
                        echoMode: TextInput.Password
                        onTextChanged: root.validateFields()
                    }
                }

                Controls.TextField {
                    id: targetFolderField
                    Kirigami.FormData.label: i18n("Target Folder:")
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    readOnly: true
                    text: configMgr ? i18n("Invoices://") + root.buildFolderName() : ""
                    font.family: "monospace"
                }

                Item {
                    Layout.fillWidth: true
                    Layout.preferredHeight: Kirigami.Units.largeSpacing
                }

                RowLayout {
                    Kirigami.FormData.isSection: false
                    Layout.fillWidth: true
                    Layout.minimumWidth: Kirigami.Units.gridUnit * 40
                    spacing: Kirigami.Units.smallSpacing

                    Item { Layout.fillWidth: true }

                    Controls.Button {
                        text: i18n("Cancel")
                        icon.name: "dialog-cancel"
                        onClicked: Qt.quit()
                    }

                    Controls.Button {
                        text: root.isEditMode ? i18n("Update Invoice Folder") : i18n("Create Invoice Folder")
                        enabled: allFieldsFilled && configMgr && configMgr.isConfigured() && !root.isBusy
                        icon.name: root.isEditMode ? "document-save" : "dialog-ok-apply"

                        onClicked: {
                            if (!configMgr.validatePin(pinField.text)) {
                                root.showPassiveNotification(i18n("Error: Invalid PIN"))
                                return
                            }

                            if (root.isEditMode) {
                                // Update existing invoice folder
                                root.isBusy = true
                                root.busyMessage = i18n("Updating invoice metadata...")

                                let folderIcon = reviewedCheckbox.checked
                                    ? configMgr.getFolderIconCheckedYes()
                                    : configMgr.getFolderIconCheckedNo()

                                let success = invoiceOps.updateInvoiceMetadata(
                                    root.editFolderPath,
                                    receiptStampField.text,
                                    dateField.text,
                                    senderField.text,
                                    invoiceNumField.text,
                                    root.extractBudgetCode(),
                                    userField.text,
                                    reviewedCheckbox.checked,
                                    commentField.text,
                                    amountField.text,
                                    folderIcon
                                )

                                root.isBusy = false
                                if (success) {
                                    root.showPassiveNotification(i18n("Invoice metadata updated successfully"))
                                } else {
                                    root.showPassiveNotification(i18n("Error: Failed to update metadata"))
                                }
                                return
                            }

                            // Create new invoice folder
                            let folderName = root.buildFolderName()
                            let targetBasePath = configMgr.getPath("invoices")

                            // Check if folder already exists !FIXME! What if a User wants to overrule the Warning?
                            if (invoiceOps.folderExists(targetBasePath, folderName)) {
                                root.showPassiveNotification(i18n("Error: Target folder already exists"))
                                return
                            }

                            // Check if invoice number already exists !FIXME! What if a User wants to overrule the Warning?
                            let invoiceNumber = invoiceNumField.text
                            if (invoiceOps.invoiceNumberExists(targetBasePath, invoiceNumber)) {
                                root.showPassiveNotification(i18n("Error: Invoice number already exists in target directory"))
                                return
                            }

                            root.isBusy = true
                            root.busyMessage = i18n("Creating invoice folder...")
                            createFolderTimer.folderName = folderName
                            createFolderTimer.start()
                        }
                    }
                }
            }

            Item {
                Layout.fillHeight: true
            }

            Controls.Label {
                text: "XRDialog v" + appVersion
                font.pointSize: 8
                color: Kirigami.Theme.disabledTextColor
                Layout.alignment: Qt.AlignHCenter
            }
        }
    }

    FileDialog {
        id: fileDialog
        title: i18n("Select Invoice File")
        nameFilters: [i18n("Invoice files (*.pdf *.xml *.png *.jpg)"), i18n("All files (*)")]
        currentFolder: StandardPaths.standardLocations(StandardPaths.HomeLocation)[0]

        onAccepted: {
            root.selectedInvoice = fileDialog.selectedFile.toString()

            // Check if file is in an existing invoice folder with metadata
            let metadata = invoiceOps.loadInvoiceMetadata(root.selectedInvoice)
            if (metadata.hasMetadata) {
                // Edit mode: populate fields from existing .directory
                root.isEditMode = true
                root.editFolderPath = metadata.folderPath

                receiptStampField.text = metadata.receiptDate || ""
                dateField.text = metadata.issueDate || ""
                senderField.text = metadata.seller || ""
                invoiceNumField.text = metadata.id || ""
                budgetCodeField.text = metadata.buyerReference || ""
                amountField.text = metadata.payableAmount || ""
                reviewedCheckbox.checked = metadata.reviewed || false
                commentField.text = metadata.note || ""

                root.showPassiveNotification(i18n("Loaded existing invoice data for editing"))
            } else {
                // New mode: reset edit state
                root.isEditMode = false
                root.editFolderPath = ""
            }

            root.validateFields()
        }

        Component.onCompleted: {
            if (configMgr && configMgr.isConfigured()) {
                let path = configMgr.getPath("incoming_invoices")
                if (path !== "") {
                    fileDialog.currentFolder = "file://" + path
                }
            }
        }
    }

    Timer {
        id: closeTimer
        interval: 1000
        repeat: false
        onTriggered: Qt.quit()
    }

    Timer {
        id: previewTimer
        interval: 50
        repeat: false
        onTriggered: {
            let cleanPath = root.selectedInvoice
            if (cleanPath.startsWith("file://")) {
                cleanPath = cleanPath.substring(7)
            }
            let tempDir = StandardPaths.writableLocation(StandardPaths.TempLocation).toString()
            if (tempDir.startsWith("file://")) {
                tempDir = tempDir.substring(7)
            }
            let previewPath = tempDir + "/xrdialog_preview.pdf"
            let success = pdfProcessor.createStampedCopy(
                cleanPath,
                tempDir,
                "xrdialog_preview",
                receiptStampField.text
            )
            root.isBusy = false
            root.busyMessage = ""
            if (success) {
                invoiceOps.openFile(previewPath)
            } else {
                root.showPassiveNotification(i18n("Preview failed - check [Stamp] and [OCR] sections in config file"))
            }
        }
    }

    Timer {
        id: createFolderTimer
        interval: 50
        repeat: false
        property string folderName: ""
        onTriggered: {
            let folderIcon = reviewedCheckbox.checked
                ? configMgr.getFolderIconCheckedYes()
                : configMgr.getFolderIconCheckedNo()
            let success = invoiceOps.createInvoiceFolder(
                root.selectedInvoice,
                configMgr.getPath("invoices"),
                receiptStampField.text,
                dateField.text,
                senderField.text,
                invoiceNumField.text,
                root.extractBudgetCode(),
                userField.text,
                reviewedCheckbox.checked,
                commentField.text,
                amountField.text,
                folderIcon
            )

            if (success) {
                root.busyMessage = i18n("Creating stamped PDF...")
                let outputFolder = configMgr.getPath("invoices") + "/" + folderName
                let originalFileName = root.selectedInvoice.split("/").pop()
                if (originalFileName.startsWith("file://")) {
                    originalFileName = originalFileName.substring(7)
                }
                let movedFilePath = outputFolder + "/" + originalFileName
                let stampedFilePath = outputFolder + "/" + folderName + ".pdf"

                let stampSuccess = pdfProcessor.createStampedCopy(
                    movedFilePath,
                    outputFolder,
                    folderName,
                    receiptStampField.text
                )
                if (stampSuccess) {
                    invoiceOps.logStampedPdf(stampedFilePath)
                    thumbGen.createStampedThumbnail(stampedFilePath)
                } else {
                    invoiceOps.logError("Stamp", "Failed to create stamped PDF")
                }

                // Create checksums for files
                let checksum1 = checksumGen.calculateChecksum(movedFilePath)
                if (checksumGen.createChecksumFile(movedFilePath)) {
                    invoiceOps.logChecksum(movedFilePath, checksum1)
                }
                let checksum2 = checksumGen.calculateChecksum(stampedFilePath)
                if (checksumGen.createChecksumFile(stampedFilePath)) {
                    invoiceOps.logChecksum(stampedFilePath, checksum2)
                }

                // Log PIN validation
                invoiceOps.logPinValidated(userField.text)
                invoiceOps.logBudgetCode(root.extractBudgetCode())
                invoiceOps.logReviewedStatus(reviewedCheckbox.checked, userField.text)

                // Create notice file
                if (commentField.text !== "") {
                    invoiceOps.createNoticeFile(outputFolder, dateField.text, commentField.text)
                }

                invoiceOps.finalizeLog()

                // Move thumbnails to Invoice Folder
                thumbGen.moveThumbnailsToFolder(outputFolder)

                // Open Folder
                if (configMgr && configMgr.openFolderAfterCreate()) {
                    invoiceOps.openFile(outputFolder)
                }

                root.isBusy = false
                root.busyMessage = ""
                root.showPassiveNotification(i18n("Invoice folder created successfully"))
                if (configMgr && configMgr.closeAfterCreate()) {
                    closeTimer.start()
                }
            } else {
                root.isBusy = false
                root.busyMessage = ""
                root.showPassiveNotification(i18n("Error: Failed to create invoice folder"))
            }
        }
    }

    onSelectedInvoiceChanged: {
        validateFields()
        if (root.selectedInvoice.toLowerCase().endsWith(".pdf")) {
            thumbGen.createThumbnail(root.selectedInvoice)
        }
    }

    Component.onCompleted: {
        // Check for metadata if started with file parameter (mimetype association)
        if (root.selectedInvoice !== "") {
            let metadata = invoiceOps.loadInvoiceMetadata(root.selectedInvoice)
            if (metadata.hasMetadata) {
                root.isEditMode = true
                root.editFolderPath = metadata.folderPath

                receiptStampField.text = metadata.receiptDate || ""
                dateField.text = metadata.issueDate || ""
                senderField.text = metadata.seller || ""
                invoiceNumField.text = metadata.id || ""
                budgetCodeField.text = metadata.buyerReference || ""
                amountField.text = metadata.payableAmount || ""
                reviewedCheckbox.checked = metadata.reviewed || false
                commentField.text = metadata.note || ""

                root.showPassiveNotification(i18n("Loaded existing invoice data for editing"))
            }
        }
        validateFields()
    }

    // Busy overlay !FIXME! Stalls if config file is corrupted - bad Idea?
    Rectangle {
        anchors.fill: parent
        color: "#80000000"
        visible: root.isBusy
        z: 999

        MouseArea {
            anchors.fill: parent
            enabled: root.isBusy
        }

        ColumnLayout {
            anchors.centerIn: parent
            spacing: Kirigami.Units.largeSpacing

            Controls.BusyIndicator {
                running: root.isBusy
                Layout.alignment: Qt.AlignHCenter
                Layout.preferredWidth: Kirigami.Units.gridUnit * 4
                Layout.preferredHeight: Kirigami.Units.gridUnit * 4
            }

            Controls.Label {
                text: root.busyMessage
                color: "white"
                font.bold: true
                Layout.alignment: Qt.AlignHCenter
                visible: root.busyMessage !== ""
            }
        }
    }
}
