#include "configmanager.h"
#include <QSettings>
#include <QDir>
#include <QFile>
#include <QTextStream>
#include <QDebug>
#include <QCryptographicHash>

ConfigManager::ConfigManager(QObject *parent)
    : QObject(parent)
    , configured(false)
    , stampFont("")
    , stampFontSize(0)
    , stampColor("")
    , stampX(-1)
    , stampY(-1)
    , stampRotation(0)
    , stampPrefix("")
    , pageWidth(0)
    , pageHeight(0)
    , ocrLanguage("")
    , folderIconCheckedYes("")
    , folderIconCheckedNo("")
    , folderIconCheckedUndefined("")
    , helpUrl("")
    , openFolderAfterCreateEnabled(false)
    , closeAfterCreateEnabled(true)
{
}

bool ConfigManager::loadConfig()
{
    configErrors.clear();
    QString configPath = QDir::homePath() + "/.config/xrdialog/config";

    QFile configFile(configPath);
    if (!configFile.exists()) {
        configErrors = "Config file not found: " + configPath;
        configured = false;
        return false;
    }

    QSettings settings(configPath, QSettings::IniFormat);

    if (settings.status() != QSettings::NoError) {
        configErrors = "Config file read error";
        configured = false;
        return false;
    }

    settings.beginGroup("Paths");
    QStringList keys = settings.allKeys();

    if (keys.isEmpty()) {
        configErrors = "No paths configured in [Paths] section";
        configured = false;
        return false;
    }

    for (const QString &key : keys) {
        paths[key] = settings.value(key).toString();
    }

    settings.endGroup();

    // Load PIN hash from secret file !FIXME! Should be replaced by suitable electronic ID soon
    QString secretPath = QDir::homePath() + "/.config/xrdialog/secret";
    QSettings secretSettings(secretPath, QSettings::IniFormat);
    secretSettings.beginGroup("Security");
    storedPinHash = secretSettings.value("pin_hash", "").toString();
    secretSettings.endGroup();

    settings.beginGroup("Stamp");
    stampFont = settings.value("font", "").toString();
    stampFontSize = settings.value("font_size", 0).toInt();
    stampColor = settings.value("color", "").toString();
    stampX = settings.value("x", -1).toInt();
    stampY = settings.value("y", -1).toInt();
    stampRotation = settings.value("rotation", 0).toInt();
    stampPrefix = settings.value("prefix", "").toString();
    pageWidth = settings.value("page_width", 0).toInt();
    pageHeight = settings.value("page_height", 0).toInt();
    settings.endGroup();

    settings.beginGroup("OCR");
    ocrLanguage = settings.value("language", "").toString();
    settings.endGroup();

    settings.beginGroup("Folder");
    folderIconCheckedYes = settings.value("icon_checked_yes", "").toString();
    folderIconCheckedNo = settings.value("icon_checked_no", "").toString();
    folderIconCheckedUndefined = settings.value("icon_checked_undefined", "").toString();
    settings.endGroup();

    settings.beginGroup("Help");
    helpUrl = settings.value("help_url", "").toString();
    settings.endGroup();

    settings.beginGroup("BudgetCode");
    budgetCodePath = settings.value("budget_code_path", "").toString();
    if (budgetCodePath.startsWith("~/")) {
        budgetCodePath = QDir::homePath() + budgetCodePath.mid(1);
    }
    settings.endGroup();

    settings.beginGroup("Behavior");
    openFolderAfterCreateEnabled = settings.value("open_folder_after_create", false).toBool();
    closeAfterCreateEnabled = settings.value("close_after_create", true).toBool();
    settings.endGroup();

    configured = true;
    loadBudgetCodes();

    // Missing configuration report to stdout
    QStringList warnings;

    // Paths
    if (!paths.contains("incoming_invoices") || paths["incoming_invoices"].isEmpty())
        warnings << "[Paths] incoming_invoices not set";
    if (!paths.contains("invoices") || paths["invoices"].isEmpty())
        warnings << "[Paths] invoices not set";

    // Security
    if (storedPinHash.isEmpty())
        warnings << "[Security] pin_hash not set in secret file";

    // Stamp
    if (stampFont.isEmpty()) warnings << "[Stamp] font not set";
    if (stampFontSize <= 0) warnings << "[Stamp] font_size not set";
    if (stampColor.isEmpty()) warnings << "[Stamp] color not set";
    if (stampX < 0) warnings << "[Stamp] x not set";
    if (stampY < 0) warnings << "[Stamp] y not set";
    if (stampPrefix.isEmpty()) warnings << "[Stamp] prefix not set";
    if (pageWidth <= 0) warnings << "[Stamp] page_width not set";
    if (pageHeight <= 0) warnings << "[Stamp] page_height not set";

    // OCR
    if (ocrLanguage.isEmpty()) warnings << "[OCR] language not set";

    // Folder
    if (folderIconCheckedYes.isEmpty()) warnings << "[Folder] icon_checked_yes not set";
    if (folderIconCheckedNo.isEmpty()) warnings << "[Folder] icon_checked_no not set";
    if (folderIconCheckedUndefined.isEmpty()) warnings << "[Folder] icon_checked_undefined not set";

    // BudgetCode
    if (budgetCodePath.isEmpty()) warnings << "[BudgetCode] budget_code_path not set";
    if (budgetCodes.isEmpty()) warnings << "[BudgetCode] No budget codes loaded (file missing or empty)";

    // Help
    if (helpUrl.isEmpty()) warnings << "[Help] help_url not set (Info button will be disabled)";

    if (!warnings.isEmpty()) {
        qWarning() << "XRDialog Configuration Warnings:";
        for (const QString &w : warnings) {
            qWarning() << "  -" << w;
        }
    }

    return true;
}

QString ConfigManager::getPath(const QString &key)
{
    return paths.value(key, "");
}

bool ConfigManager::isConfigured() const
{
    return configured;
}

QStringList ConfigManager::getBudgetCodes()
{
    return budgetCodes;
}

bool ConfigManager::validatePin(const QString &pin) const
{
    if (storedPinHash.isEmpty() || pin.isEmpty()) {
        return false;
    }
    QString inputHash = QString(QCryptographicHash::hash(
        pin.toUtf8(), QCryptographicHash::Sha256).toHex());
    return inputHash == storedPinHash;
}

QString ConfigManager::getConfigErrors() const
{
    return configErrors;
}

bool ConfigManager::checkDirectories()
{
    QStringList errors;

    for (auto it = paths.constBegin(); it != paths.constEnd(); ++it) {
        QString path = it.value();
        QDir dir(path);

        if (!dir.exists()) {
            errors.append("Directory does not exist: " + path + " (" + it.key() + ")");
            continue;
        }

        QFile testFile(path + "/.write_test");
        if (!testFile.open(QIODevice::WriteOnly)) {
            errors.append("Directory not writable: " + path + " (" + it.key() + ")");
        } else {
            testFile.close();
            testFile.remove();
        }
    }

    if (!errors.isEmpty()) {
        configErrors = errors.join("\n");
        return false;
    }

    return true;
}

void ConfigManager::loadBudgetCodes()
{
    QFile file(budgetCodePath);

    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        return;
    }

    QTextStream in(&file);
    while (!in.atEnd()) {
        QString line = in.readLine().trimmed();
        if (!line.isEmpty()) {
            budgetCodes.append(line);
        }
    }

    file.close();
}

QString ConfigManager::getStampFont() const
{
    return stampFont;
}

int ConfigManager::getStampFontSize() const
{
    return stampFontSize;
}

QString ConfigManager::getStampColor() const
{
    return stampColor;
}

int ConfigManager::getStampX() const
{
    return stampX;
}

int ConfigManager::getStampY() const
{
    return stampY;
}

int ConfigManager::getStampRotation() const
{
    return stampRotation;
}

QString ConfigManager::getStampPrefix() const
{
    return stampPrefix;
}

int ConfigManager::getPageWidth() const
{
    return pageWidth;
}

int ConfigManager::getPageHeight() const
{
    return pageHeight;
}

QString ConfigManager::getOcrLanguage() const
{
    return ocrLanguage;
}

bool ConfigManager::isStampConfigured() const
{
    return !stampFont.isEmpty() &&
           stampFontSize > 0 &&
           !stampColor.isEmpty() &&
           stampX >= 0 &&
           stampY >= 0 &&
           !stampPrefix.isEmpty() &&
           pageWidth > 0 &&
           pageHeight > 0;
}

QString ConfigManager::getStampConfigErrors() const
{
    QStringList errors;
    if (stampFont.isEmpty()) errors << "[Stamp] font not set";
    if (stampFontSize <= 0) errors << "[Stamp] font_size not set";
    if (stampColor.isEmpty()) errors << "[Stamp] color not set";
    if (stampX < 0) errors << "[Stamp] x not set";
    if (stampY < 0) errors << "[Stamp] y not set";
    if (stampPrefix.isEmpty()) errors << "[Stamp] prefix not set";
    if (pageWidth <= 0) errors << "[Stamp] page_width not set";
    if (pageHeight <= 0) errors << "[Stamp] page_height not set";
    return errors.join("\n");
}

bool ConfigManager::isOcrConfigured() const
{
    return !ocrLanguage.isEmpty();
}

QString ConfigManager::getFolderIconCheckedYes() const
{
    return folderIconCheckedYes;
}

QString ConfigManager::getFolderIconCheckedNo() const
{
    return folderIconCheckedNo;
}

QString ConfigManager::getFolderIconCheckedUndefined() const
{
    return folderIconCheckedUndefined;
}

bool ConfigManager::isFolderIconsConfigured() const
{
    return !folderIconCheckedYes.isEmpty() &&
           !folderIconCheckedNo.isEmpty() &&
           !folderIconCheckedUndefined.isEmpty();
}

QString ConfigManager::getFolderIconConfigErrors() const
{
    QStringList errors;
    if (folderIconCheckedYes.isEmpty()) errors << "[Folder] icon_checked_yes not set";
    if (folderIconCheckedNo.isEmpty()) errors << "[Folder] icon_checked_no not set";
    if (folderIconCheckedUndefined.isEmpty()) errors << "[Folder] icon_checked_undefined not set";
    return errors.join("\n");
}

QString ConfigManager::getHelpUrl() const
{
    return helpUrl;
}

bool ConfigManager::openFolderAfterCreate() const
{
    return openFolderAfterCreateEnabled;
}

bool ConfigManager::closeAfterCreate() const
{
    return closeAfterCreateEnabled;
}
