#include "invoiceoperations.h"
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QProcess>
#include <QDateTime>
#include <QSettings>
#include <unistd.h>
#include <pwd.h>

InvoiceOperations::InvoiceOperations(QObject *parent)
    : QObject(parent)
{
}

bool InvoiceOperations::createInvoiceFolder(
    const QString &invoiceFile,
    const QString &targetBasePath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    if (invoiceFile.isEmpty() || targetBasePath.isEmpty() ||
        receiptStamp.isEmpty() || date.isEmpty() || sender.isEmpty() ||
        invoiceNumber.isEmpty() || budgetCode.isEmpty() ||
        user.isEmpty()) {
        workflowLogger.logError("Validation", "Required fields missing");
        return false;
    }

    QString folderName = buildFolderName(date, sender, invoiceNumber,
                                          budgetCode, user, reviewed, comment);
    QString targetPath = targetBasePath + "/" + folderName;

    workflowLogger.setLogPath(targetPath);

    QDir dir;
    if (!dir.mkpath(targetPath)) {
        workflowLogger.logError("Folder", "Failed to create: " + targetPath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFolderCreated(targetPath);

    QString cleanInvoiceFile = invoiceFile;
    if (cleanInvoiceFile.startsWith("file://")) {
        cleanInvoiceFile = cleanInvoiceFile.mid(7);
    }

    QFile file(cleanInvoiceFile);
    if (!file.exists()) {
        workflowLogger.logError("Source", "File not found: " + cleanInvoiceFile);
        workflowLogger.writeLog();
        return false;
    }

    QFileInfo fileInfo(cleanInvoiceFile);
    QString targetFilePath = targetPath + "/" + fileInfo.fileName();

    if (!file.rename(targetFilePath)) {
        workflowLogger.logError("Move", "Failed to move file to: " + targetFilePath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFileMoved(cleanInvoiceFile, targetFilePath);

    createDirectoryFile(targetPath, receiptStamp, date, sender, invoiceNumber, budgetCode,
                        user, reviewed, comment, amount, folderIcon);

    return true;
}

void InvoiceOperations::createDirectoryFile(
    const QString &folderPath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    QString directoryFilePath = folderPath + "/.directory";
    QFile directoryFile(directoryFilePath);

    if (!directoryFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return;
    }

    QTextStream out(&directoryFile);

    // .directory 
    out << "[Desktop Entry]\n";
    out << "Type=Directory\n";
    out << "Icon=" << folderIcon << "\n";
    out << "Comment=Invoice: " << invoiceNumber << " from " << sender << "\n";

    // [invoice] section
    out << "\n[invoice]\n";
    out << "FormatVersion=1.0\n";
    out << "ReceiptDate=" << receiptStamp << "\n";
    out << "IssueDate=" << date << "\n";
    out << "Seller=" << sender << "\n";
    out << "ID=" << invoiceNumber << "\n";
    out << "BuyerReference=" << budgetCode << "\n";
    if (!amount.isEmpty()) {
        out << "PayableAmount=" << amount << "\n";
        out << "PayableCurrency=EUR\n";
    }

    // [p]rocessing] section
    out << "\n[processing]\n";
    out << "CreatedAt=" << QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
    out << "ProcessedBy=" << user << "\n";
    out << "Reviewed=" << (reviewed ? "true" : "false") << "\n";
    if (!comment.isEmpty()) {
        // !FIXME! Maybe in *.txt file only in future?
        out << "Note=" << comment << "\n";
    }

    directoryFile.close();
}

QString InvoiceOperations::getCurrentUser()
{
    QString user = qgetenv("USER");

    if (user.isEmpty()) {
        struct passwd *pw = getpwuid(getuid());
        if (pw != nullptr) {
            user = QString::fromUtf8(pw->pw_name);
        }
    }

    return user;
}

QString InvoiceOperations::buildFolderName(
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool /* reviewed */,
    const QString & /* comment */)
{
    QString cleanSender = QString(sender).replace(" ", "_");
    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "_");

    return QString("%1_%2_%3_%4_%5")
        .arg(date)
        .arg(cleanSender)
        .arg(cleanInvoiceNumber)
        .arg(budgetCode)
        .arg(user);
}

void InvoiceOperations::openFile(const QString &filePath)
{
    QProcess::startDetached("xdg-open", QStringList() << filePath);
}

void InvoiceOperations::logChecksum(const QString &filePath, const QString &checksum)
{
    workflowLogger.logChecksumCreated(filePath, checksum);
}

void InvoiceOperations::logStampedPdf(const QString &filePath)
{
    workflowLogger.logStampedPdfCreated(filePath);
}

void InvoiceOperations::logPinValidated(const QString &user)
{
    workflowLogger.logPinValidated(user);
}

void InvoiceOperations::logReviewedStatus(bool reviewed, const QString &user)
{
    workflowLogger.logReviewedStatus(reviewed, user);
}

void InvoiceOperations::logBudgetCode(const QString &budgetCode)
{
    workflowLogger.logBudgetCode(budgetCode);
}

void InvoiceOperations::logError(const QString &operation, const QString &message)
{
    workflowLogger.logError(operation, message);
}

void InvoiceOperations::finalizeLog()
{
    workflowLogger.writeLog();
}

bool InvoiceOperations::folderExists(const QString &targetBasePath, const QString &folderName)
{
    QString targetPath = targetBasePath + "/" + folderName;
    QDir dir(targetPath);
    return dir.exists();
}

bool InvoiceOperations::invoiceNumberExists(const QString &targetBasePath, const QString &invoiceNumber)
{
    if (invoiceNumber.isEmpty() || targetBasePath.isEmpty()) {
        return false;
    }

    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "_");
    QDir dir(targetBasePath);

    if (!dir.exists()) {
        return false;
    }

    // Read invoice ID from .directory files instead of parsing folder names
    // This is more reliable since folder names may contain underscores in sender names (!FIXME! Test-> Does it **WORK**?)
    QStringList entries = dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot);
    for (const QString &entry : entries) {
        QString directoryFile = targetBasePath + "/" + entry + "/.directory";
        QSettings settings(directoryFile, QSettings::IniFormat);
        settings.beginGroup("invoice");
        QString existingId = settings.value("ID", "").toString();
        settings.endGroup();

        if (!existingId.isEmpty() && existingId == cleanInvoiceNumber) {
            return true;
        }
    }

    return false;
}

bool InvoiceOperations::createNoticeFile(const QString &folderPath, const QString &date, const QString &noticeText)
{
    if (noticeText.isEmpty()) {
        return true; // No *.txt for create create
    }

    QString noticeFilePath = folderPath + "/" + date + "_Notice.txt";
    QFile noticeFile(noticeFilePath);

    if (!noticeFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        workflowLogger.logError("Notice", "Failed to create notice file: " + noticeFilePath);
        return false;
    }

    QTextStream out(&noticeFile);
    out << noticeText << "\n";
    noticeFile.close();

    return true;
}

QVariantMap InvoiceOperations::loadInvoiceMetadata(const QString &filePath)
{
    QVariantMap result;

    QString cleanPath = filePath;
    if (cleanPath.startsWith("file://")) {
        cleanPath = cleanPath.mid(7);
    }

    QFileInfo fileInfo(cleanPath);
    if (!fileInfo.exists()) {
        return result;
    }

    // Get parent directory and check for .directory file
    QString parentDir = fileInfo.absolutePath();
    QString directoryFile = parentDir + "/.directory";

    if (!QFile::exists(directoryFile)) {
        return result;
    }

    QSettings settings(directoryFile, QSettings::IniFormat);

    // Check if this is an invoice directory (has [invoice] section)
    settings.beginGroup("invoice");
    if (!settings.contains("ID")) {
        settings.endGroup();
        return result;
    }

    // Load invoice data
    result["receiptDate"] = settings.value("ReceiptDate", "").toString();
    result["issueDate"] = settings.value("IssueDate", "").toString();
    result["seller"] = settings.value("Seller", "").toString();
    result["id"] = settings.value("ID", "").toString();
    result["buyerReference"] = settings.value("BuyerReference", "").toString();
    result["payableAmount"] = settings.value("PayableAmount", "").toString();
    settings.endGroup();

    // Load processing data
    settings.beginGroup("processing");
    result["processedBy"] = settings.value("ProcessedBy", "").toString();
    result["reviewed"] = settings.value("Reviewed", "false").toString() == "true";
    result["note"] = settings.value("Note", "").toString();
    settings.endGroup();

    // Store the folder path for potential update operations
    result["folderPath"] = parentDir;
    result["hasMetadata"] = true;

    return result;
}

bool InvoiceOperations::updateInvoiceMetadata(
    const QString &folderPath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    QString directoryFilePath = folderPath + "/.directory";

    if (!QFile::exists(directoryFilePath)) {
        return false;
    }

    QFile directoryFile(directoryFilePath);
    if (!directoryFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return false;
    }

    QTextStream out(&directoryFile);

    // Desktop Entry section (freedesktop.org standard -> FIX: KDE Standard? -> Should be readable by interested applications...)
    out << "[Desktop Entry]\n";
    out << "Type=Directory\n";
    out << "Icon=" << folderIcon << "\n";
    out << "Comment=Invoice: " << invoiceNumber << " from " << sender << "\n";

    // Invoice data section
    out << "\n[invoice]\n";
    out << "FormatVersion=1.0\n";
    out << "ReceiptDate=" << receiptStamp << "\n";
    out << "IssueDate=" << date << "\n";
    out << "Seller=" << sender << "\n";
    out << "ID=" << invoiceNumber << "\n";
    out << "BuyerReference=" << budgetCode << "\n";
    if (!amount.isEmpty()) {
        out << "PayableAmount=" << amount << "\n";
        out << "PayableCurrency=EUR\n";
    }

    // Processing section
    out << "\n[processing]\n";
    out << "UpdatedAt=" << QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
    out << "ProcessedBy=" << user << "\n";
    out << "Reviewed=" << (reviewed ? "true" : "false") << "\n";
    if (!comment.isEmpty()) {
        out << "Note=" << comment << "\n";
    }

    directoryFile.close();

    // Append to existing workflow log
    workflowLogger.appendToLog(folderPath, user);

    return true;
}
