#include "invoiceoperations.h"
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QProcess>
#include <QDateTime>
#include <QSettings>
#include <unistd.h>
#include <pwd.h>

InvoiceOperations::InvoiceOperations(QObject *parent)
    : QObject(parent)
{
}

bool InvoiceOperations::createInvoiceFolder(
    const QString &invoiceFile,
    const QString &targetBasePath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    if (invoiceFile.isEmpty() || targetBasePath.isEmpty() ||
        receiptStamp.isEmpty() || date.isEmpty() || sender.isEmpty() ||
        invoiceNumber.isEmpty() || budgetCode.isEmpty() ||
        user.isEmpty()) {
        workflowLogger.logError("Validation", "Required fields missing");
        return false;
    }

    QString folderName = buildFolderName(date, sender, invoiceNumber,
                                          budgetCode, user, reviewed, comment);
    QString targetPath = targetBasePath + "/" + folderName;

    workflowLogger.setLogPath(targetPath);

    QDir dir;
    if (!dir.mkpath(targetPath)) {
        workflowLogger.logError("Folder", "Failed to create: " + targetPath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFolderCreated(targetPath);

    QString cleanInvoiceFile = invoiceFile;
    if (cleanInvoiceFile.startsWith("file://")) {
        cleanInvoiceFile = cleanInvoiceFile.mid(7);
    }

    QFile file(cleanInvoiceFile);
    if (!file.exists()) {
        workflowLogger.logError("Source", "File not found: " + cleanInvoiceFile);
        workflowLogger.writeLog();
        return false;
    }

    QFileInfo fileInfo(cleanInvoiceFile);
    QString targetFilePath = targetPath + "/" + fileInfo.fileName();

    if (!file.rename(targetFilePath)) {
        workflowLogger.logError("Move", "Failed to move file to: " + targetFilePath);
        workflowLogger.writeLog();
        return false;
    }
    workflowLogger.logFileMoved(cleanInvoiceFile, targetFilePath);

    createDirectoryFile(targetPath, receiptStamp, date, sender, invoiceNumber, budgetCode,
                        user, reviewed, comment, amount, folderIcon);

    return true;
}

void InvoiceOperations::createDirectoryFile(
    const QString &folderPath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    QString directoryFilePath = folderPath + "/.directory";
    QFile directoryFile(directoryFilePath);

    if (!directoryFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return;
    }

    QTextStream out(&directoryFile);

    // [Desktop Entry] - freedesktop.org standard fields
    out << "[Desktop Entry]\n";
    out << "Type=Directory\n";
    out << "Name=" << invoiceNumber << " - " << sender << "\n";
    out << "Name[de]=" << invoiceNumber << " - " << sender << "\n";
    out << "GenericName=Invoice\n";
    out << "GenericName[de]=Eingangsrechnung\n";
    out << "Comment=Invoice " << invoiceNumber << " from " << sender << " (" << receiptStamp << ")\n";
    out << "Comment[de]=Rechnung " << invoiceNumber << " von " << sender << " (" << receiptStamp << ")\n";
    out << "Icon=" << folderIcon << "\n";
    out << "Keywords=" << sender << ";" << invoiceNumber << ";" << budgetCode << ";Invoice;Rechnung;\n";

    // [X-XRDialog-Invoice] - vendor-specific invoice data
    out << "\n[X-XRDialog-Invoice]\n";
    out << "FormatVersion=1.0\n";
    out << "ReceiptDate=" << receiptStamp << "\n";
    out << "IssueDate=" << date << "\n";
    out << "Seller=" << sender << "\n";
    out << "ID=" << invoiceNumber << "\n";
    out << "BuyerReference=" << budgetCode << "\n";
    if (!amount.isEmpty()) {
        out << "PayableAmount=" << amount << "\n";
        out << "PayableCurrency=EUR\n";
    }

    // [X-XRDialog-Processing] - vendor-specific processing data
    out << "\n[X-XRDialog-Processing]\n";
    out << "CreatedAt=" << QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
    out << "ProcessedBy=" << user << "\n";
    out << "Reviewed=" << (reviewed ? "true" : "false") << "\n";
    if (!comment.isEmpty()) {
        out << "Note=" << comment << "\n";
    }

    directoryFile.close();
}

QString InvoiceOperations::getCurrentUser()
{
    QString user = qgetenv("USER");

    if (user.isEmpty()) {
        struct passwd *pw = getpwuid(getuid());
        if (pw != nullptr) {
            user = QString::fromUtf8(pw->pw_name);
        }
    }

    return user;
}

QString InvoiceOperations::buildFolderName(
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool /* reviewed */,
    const QString & /* comment */)
{
    QString cleanSender = QString(sender).replace(" ", "-").replace("/", "-").replace("\\", "-");
    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "-").replace("/", "-").replace("\\", "-");

    return QString("%1_%2_%3_%4_%5")
        .arg(date)
        .arg(cleanSender)
        .arg(cleanInvoiceNumber)
        .arg(budgetCode)
        .arg(user);
}

void InvoiceOperations::openFile(const QString &filePath)
{
    QProcess::startDetached("xdg-open", QStringList() << filePath);
}

void InvoiceOperations::logChecksum(const QString &filePath, const QString &checksum)
{
    workflowLogger.logChecksumCreated(filePath, checksum);
}

void InvoiceOperations::logStampedPdf(const QString &filePath)
{
    workflowLogger.logStampedPdfCreated(filePath);
}

void InvoiceOperations::logPinValidated(const QString &user)
{
    workflowLogger.logPinValidated(user);
}

void InvoiceOperations::logReviewedStatus(bool reviewed, const QString &user)
{
    workflowLogger.logReviewedStatus(reviewed, user);
}

void InvoiceOperations::logBudgetCode(const QString &budgetCode)
{
    workflowLogger.logBudgetCode(budgetCode);
}

void InvoiceOperations::logError(const QString &operation, const QString &message)
{
    workflowLogger.logError(operation, message);
}

void InvoiceOperations::finalizeLog()
{
    workflowLogger.writeLog();
}

bool InvoiceOperations::folderExists(const QString &targetBasePath, const QString &folderName)
{
    QString targetPath = targetBasePath + "/" + folderName;
    QDir dir(targetPath);
    return dir.exists();
}

bool InvoiceOperations::invoiceNumberExists(const QString &targetBasePath, const QString &invoiceNumber)
{
    if (invoiceNumber.isEmpty() || targetBasePath.isEmpty()) {
        return false;
    }

    QString cleanInvoiceNumber = QString(invoiceNumber).replace(" ", "-");
    QDir dir(targetBasePath);

    if (!dir.exists()) {
        return false;
    }

    // Read invoice ID from .directory files instead of parsing folder names
    QStringList entries = dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot);
    for (const QString &entry : entries) {
        QString directoryFile = targetBasePath + "/" + entry + "/.directory";
        QSettings settings(directoryFile, QSettings::IniFormat);
        settings.beginGroup("X-XRDialog-Invoice");
        QString existingId = settings.value("ID", "").toString();
        settings.endGroup();

        if (!existingId.isEmpty() && existingId == cleanInvoiceNumber) {
            return true;
        }
    }

    return false;
}

bool InvoiceOperations::createNoticeFile(const QString &folderPath, const QString &date, const QString &noticeText)
{
    if (noticeText.isEmpty()) {
        return true; // No *.txt for create create
    }

    QString noticeFilePath = folderPath + "/" + date + "_Notice.txt";
    QFile noticeFile(noticeFilePath);

    if (!noticeFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        workflowLogger.logError("Notice", "Failed to create notice file: " + noticeFilePath);
        return false;
    }

    QTextStream out(&noticeFile);
    out << noticeText << "\n";
    noticeFile.close();

    return true;
}

QVariantMap InvoiceOperations::loadInvoiceMetadata(const QString &filePath)
{
    QVariantMap result;

    QString cleanPath = filePath;
    if (cleanPath.startsWith("file://")) {
        cleanPath = cleanPath.mid(7);
    }

    QFileInfo fileInfo(cleanPath);
    if (!fileInfo.exists()) {
        return result;
    }

    // Get parent directory and check for .directory file
    QString parentDir = fileInfo.absolutePath();
    QString directoryFile = parentDir + "/.directory";

    if (!QFile::exists(directoryFile)) {
        return result;
    }

    QSettings settings(directoryFile, QSettings::IniFormat);

    // Check if this is an invoice directory (has [X-XRDialog-Invoice] section)
    settings.beginGroup("X-XRDialog-Invoice");
    if (!settings.contains("ID")) {
        settings.endGroup();
        return result;
    }

    // Load invoice data
    result["receiptDate"] = settings.value("ReceiptDate", "").toString();
    result["issueDate"] = settings.value("IssueDate", "").toString();
    result["seller"] = settings.value("Seller", "").toString();
    result["id"] = settings.value("ID", "").toString();
    result["buyerReference"] = settings.value("BuyerReference", "").toString();
    result["payableAmount"] = settings.value("PayableAmount", "").toString();
    settings.endGroup();

    // Load processing data
    settings.beginGroup("X-XRDialog-Processing");
    result["processedBy"] = settings.value("ProcessedBy", "").toString();
    result["reviewed"] = settings.value("Reviewed", "false").toString() == "true";
    result["note"] = settings.value("Note", "").toString();
    settings.endGroup();

    // Store the folder path for potential update operations
    result["folderPath"] = parentDir;
    result["hasMetadata"] = true;

    return result;
}

bool InvoiceOperations::updateInvoiceMetadata(
    const QString &folderPath,
    const QString &receiptStamp,
    const QString &date,
    const QString &sender,
    const QString &invoiceNumber,
    const QString &budgetCode,
    const QString &user,
    bool reviewed,
    const QString &comment,
    const QString &amount,
    const QString &folderIcon)
{
    QString directoryFilePath = folderPath + "/.directory";

    if (!QFile::exists(directoryFilePath)) {
        return false;
    }

    // Load old values for change logging
    QSettings oldSettings(directoryFilePath, QSettings::IniFormat);
    QStringList changes;

    oldSettings.beginGroup("X-XRDialog-Invoice");
    QString oldReceiptDate = oldSettings.value("ReceiptDate", "").toString();
    QString oldIssueDate = oldSettings.value("IssueDate", "").toString();
    QString oldSeller = oldSettings.value("Seller", "").toString();
    QString oldId = oldSettings.value("ID", "").toString();
    QString oldBudgetCode = oldSettings.value("BuyerReference", "").toString();
    QString oldAmount = oldSettings.value("PayableAmount", "").toString();
    oldSettings.endGroup();

    oldSettings.beginGroup("X-XRDialog-Processing");
    bool oldReviewed = oldSettings.value("Reviewed", "false").toString() == "true";
    QString oldNote = oldSettings.value("Note", "").toString();
    oldSettings.endGroup();

    // Detect changes
    if (oldReceiptDate != receiptStamp) changes << QString("ReceiptDate: %1 -> %2").arg(oldReceiptDate, receiptStamp);
    if (oldIssueDate != date) changes << QString("IssueDate: %1 -> %2").arg(oldIssueDate, date);
    if (oldSeller != sender) changes << QString("Seller: %1 -> %2").arg(oldSeller, sender);
    if (oldId != invoiceNumber) changes << QString("ID: %1 -> %2").arg(oldId, invoiceNumber);
    if (oldBudgetCode != budgetCode) changes << QString("BuyerReference: %1 -> %2").arg(oldBudgetCode, budgetCode);
    if (oldAmount != amount) changes << QString("PayableAmount: %1 -> %2").arg(oldAmount, amount);
    if (oldReviewed != reviewed) changes << QString("Reviewed: %1 -> %2").arg(oldReviewed ? "true" : "false", reviewed ? "true" : "false");
    if (oldNote != comment) changes << QString("Note: %1 -> %2").arg(oldNote, comment);

    // Write new values
    QFile directoryFile(directoryFilePath);
    if (!directoryFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return false;
    }

    QTextStream out(&directoryFile);

    // [Desktop Entry] - freedesktop.org standard fields
    out << "[Desktop Entry]\n";
    out << "Type=Directory\n";
    out << "Name=" << invoiceNumber << " - " << sender << "\n";
    out << "Name[de]=" << invoiceNumber << " - " << sender << "\n";
    out << "GenericName=Invoice\n";
    out << "GenericName[de]=Eingangsrechnung\n";
    out << "Comment=Invoice " << invoiceNumber << " from " << sender << " (" << receiptStamp << ")\n";
    out << "Comment[de]=Rechnung " << invoiceNumber << " von " << sender << " (" << receiptStamp << ")\n";
    out << "Icon=" << folderIcon << "\n";
    out << "Keywords=" << sender << ";" << invoiceNumber << ";" << budgetCode << ";Invoice;Rechnung;\n";

    // [X-XRDialog-Invoice] - vendor-specific invoice data
    out << "\n[X-XRDialog-Invoice]\n";
    out << "FormatVersion=1.0\n";
    out << "ReceiptDate=" << receiptStamp << "\n";
    out << "IssueDate=" << date << "\n";
    out << "Seller=" << sender << "\n";
    out << "ID=" << invoiceNumber << "\n";
    out << "BuyerReference=" << budgetCode << "\n";
    if (!amount.isEmpty()) {
        out << "PayableAmount=" << amount << "\n";
        out << "PayableCurrency=EUR\n";
    }

    // [X-XRDialog-Processing] - vendor-specific processing data
    out << "\n[X-XRDialog-Processing]\n";
    out << "UpdatedAt=" << QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
    out << "ProcessedBy=" << user << "\n";
    out << "Reviewed=" << (reviewed ? "true" : "false") << "\n";
    if (!comment.isEmpty()) {
        out << "Note=" << comment << "\n";
    }

    directoryFile.close();

    // Append changes to existing workflow log
    workflowLogger.appendToLogWithChanges(folderPath, user, changes);

    return true;
}

QString InvoiceOperations::getNextVersionedFilename(const QString &folderPath, const QString &baseName)
{
    QString baseFilePath = folderPath + "/" + baseName + ".pdf";

    // If base file doesn't exist, return it
    if (!QFile::exists(baseFilePath)) {
        return baseName;
    }

    // Find the next available version number
    int version = 2;
    while (true) {
        QString versionedName = QString("%1_v%2").arg(baseName).arg(version);
        QString versionedPath = folderPath + "/" + versionedName + ".pdf";

        if (!QFile::exists(versionedPath)) {
            return versionedName;
        }

        version++;

        // Safety limit to prevent infinite loop !FIXME! Better solution needed
        if (version > 999) {
            return QString("%1_v%2").arg(baseName).arg(QDateTime::currentDateTime().toSecsSinceEpoch());
        }
    }
}
