#include "pdfprocessor.h"
#include "configmanager.h"
#include <QProcess>
#include <QFile>
#include <QDir>
#include <QTextStream>
#include <QStandardPaths>
#include <QPageSize>
#include <QColor>

PdfProcessor::PdfProcessor(QObject *parent)
    : QObject(parent)
    , configMgr(nullptr)
{
}

void PdfProcessor::setConfigManager(ConfigManager *config)
{
    configMgr = config;
}

bool PdfProcessor::isAvailable() const
{
    QProcess process;
    process.start("pdftk", QStringList() << "--version");
    process.waitForFinished(3000);
    return process.exitCode() == 0;
}

bool PdfProcessor::isOcrAvailable() const
{
    if (!configMgr || !configMgr->isOcrConfigured()) {
        return false;
    }

    QProcess process;
    process.start("ocrmypdf", QStringList() << "--version");
    process.waitForFinished(3000);
    return process.exitCode() == 0;
}

bool PdfProcessor::applyOcr(const QString &inputPdf, const QString &outputPdf)
{
    if (!configMgr || !configMgr->isOcrConfigured()) {
        return false;
    }

    QString language = configMgr->getOcrLanguage();

    QProcess process;
    QStringList args;
    args << "--language" << language
         << "--skip-text"
         << "--output-type" << "pdfa-2"
         << inputPdf << outputPdf;

    process.start("ocrmypdf", args);
    process.waitForFinished(120000);

    return process.exitCode() == 0 && QFile::exists(outputPdf);
}

bool PdfProcessor::createStampPdf(const QString &stampPath, const QString &receiptDate)
{
    if (!configMgr || !configMgr->isStampConfigured()) {
        return false;
    }

    QString font = configMgr->getStampFont();
    int fontSize = configMgr->getStampFontSize();
    QString colorHex = configMgr->getStampColor();
    int stampX = configMgr->getStampX();
    int stampY = configMgr->getStampY();
    int rotation = configMgr->getStampRotation();
    QString prefix = configMgr->getStampPrefix();
    int pageWidth = configMgr->getPageWidth();
    int pageHeight = configMgr->getPageHeight();

    QColor color(colorHex);
    double r = color.redF();
    double g = color.greenF();
    double b = color.blueF();

    QString tempDir = QStandardPaths::writableLocation(QStandardPaths::TempLocation);
    QString psPath = tempDir + "/xrdialog_stamp.ps";

    QFile psFile(psPath);
    if (!psFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return false;
    }

    QTextStream out(&psFile);
    out << "%!PS-Adobe-3.0\n";
    out << QString("%%%%BoundingBox: 0 0 %1 %2\n").arg(pageWidth).arg(pageHeight);
    out << "%%EndComments\n";
    out << QString("/%1 findfont %2 scalefont setfont\n").arg(font).arg(fontSize);
    out << QString("%1 %2 %3 setrgbcolor\n").arg(r, 0, 'f', 3).arg(g, 0, 'f', 3).arg(b, 0, 'f', 3);
    out << "gsave\n";
    out << QString("%1 %2 translate\n").arg(stampX).arg(stampY);
    out << QString("%1 rotate\n").arg(rotation);
    out << "0 0 moveto\n";
    out << QString("(%1 %2) show\n").arg(prefix).arg(receiptDate);
    out << "grestore\n";
    out << "showpage\n";
    psFile.close();

    QProcess process;
    QStringList ps2pdfArgs;
    ps2pdfArgs << "-dNOSAFER" << psPath << stampPath;
    process.start("ps2pdf", ps2pdfArgs);
    process.waitForFinished(10000);

    int exitCode = process.exitCode();
    QFile::remove(psPath);

    return exitCode == 0 && QFile::exists(stampPath);
}

bool PdfProcessor::wasOcrApplied() const
{
    return lastOcrApplied;
}

bool PdfProcessor::createStampedCopy(
    const QString &inputPdf,
    const QString &outputFolder,
    const QString &outputName,
    const QString &receiptDate)
{
    lastOcrApplied = false;

    if (inputPdf.isEmpty() || outputFolder.isEmpty() || outputName.isEmpty()) {
        return false;
    }

    QString cleanInput = inputPdf;
    if (cleanInput.startsWith("file://")) {
        cleanInput = cleanInput.mid(7);
    }

    if (!QFile::exists(cleanInput)) {
        return false;
    }

    if (!cleanInput.toLower().endsWith(".pdf")) {
        return false;
    }

    QString tempDir = QStandardPaths::writableLocation(QStandardPaths::TempLocation);
    QString stampPath = tempDir + "/xrdialog_stamp.pdf";
    QString stampedPath = tempDir + "/xrdialog_stamped.pdf";
    QString outputPath = outputFolder + "/" + outputName + ".pdf";

    QFile::remove(stampPath);
    QFile::remove(stampedPath);
    QFile::remove(outputPath);

    if (!createStampPdf(stampPath, receiptDate)) {
        if (isOcrAvailable()) {
            lastOcrApplied = applyOcr(cleanInput, outputPath);
            return lastOcrApplied;
        }
        return QFile::copy(cleanInput, outputPath);
    }

    QProcess process;
    QStringList args;
    args << cleanInput
         << "stamp" << stampPath
         << "output" << stampedPath;

    process.start("pdftk", args);
    process.waitForFinished(30000);

    int pdftkExit = process.exitCode();
    QFile::remove(stampPath);

    if (pdftkExit != 0) {
        if (isOcrAvailable()) {
            lastOcrApplied = applyOcr(cleanInput, outputPath);
            return lastOcrApplied;
        }
        return QFile::copy(cleanInput, outputPath);
    }

    if (isOcrAvailable()) {
        lastOcrApplied = applyOcr(stampedPath, outputPath);
        if (lastOcrApplied) {
            QFile::remove(stampedPath);
            return true;
        }
    }

    if (QFile::exists(stampedPath)) {
        bool moved = QFile::rename(stampedPath, outputPath);
        if (!moved) {
            if (QFile::copy(stampedPath, outputPath)) {
                QFile::remove(stampedPath);
            }
        }
    }

    return QFile::exists(outputPath);
}
