#include "workflowlogger.h"
#include <QFile>
#include <QTextStream>
#include <QDateTime>

WorkflowLogger::WorkflowLogger(QObject *parent)
    : QObject(parent)
{
}

void WorkflowLogger::setLogPath(const QString &folderPath)
{
    logPath = folderPath + "/.workflow.log";
    logEntries.clear();
    logEntries << "=== XRDialog Workflow Log ===";
    logEntries << "Started: " + QDateTime::currentDateTime().toString(Qt::ISODate);
    logEntries << "";
}

QString WorkflowLogger::timestamp()
{
    return QDateTime::currentDateTime().toString("hh:mm:ss");
}

void WorkflowLogger::logFolderCreated(const QString &folderPath)
{
    logEntries << timestamp() + " [FOLDER] Created: " + folderPath;
}

void WorkflowLogger::logFileMoved(const QString &sourcePath, const QString &targetPath)
{
    logEntries << timestamp() + " [MOVE] " + sourcePath;
    logEntries << "         -> " + targetPath;
}

void WorkflowLogger::logChecksumCreated(const QString &filePath, const QString &checksum)
{
    logEntries << timestamp() + " [SHA256] " + filePath;
    logEntries << "         " + checksum;
}

void WorkflowLogger::logStampedPdfCreated(const QString &filePath)
{
    logEntries << timestamp() + " [STAMP] Created: " + filePath;
}

void WorkflowLogger::logPinValidated(const QString &user)
{
    logEntries << timestamp() + " [AUTH] PIN validated for user: " + user;
}

void WorkflowLogger::logReviewedStatus(bool reviewed, const QString &user)
{
    if (reviewed) {
        logEntries << timestamp() + " [REVIEW] Invoice marked as reviewed by: " + user;
    } else {
        logEntries << timestamp() + " [REVIEW] Invoice NOT reviewed (pending)";
    }
}

void WorkflowLogger::logBudgetCode(const QString &budgetCode)
{
    logEntries << timestamp() + " [BUDGET] BuyerReference: " + budgetCode;
}

void WorkflowLogger::logError(const QString &operation, const QString &message)
{
    logEntries << timestamp() + " [ERROR] " + operation + ": " + message;
}

void WorkflowLogger::logMetadataUpdated(const QString &user)
{
    logEntries << timestamp() + " [UPDATE] Metadata updated by: " + user;
}

void WorkflowLogger::appendToLog(const QString &folderPath, const QString &user)
{
    QString appendLogPath = folderPath + "/.workflow.log";

    QFile file(appendLogPath);
    if (file.open(QIODevice::Append | QIODevice::Text)) {
        QTextStream out(&file);
        out << "\n";
        out << "=== Metadata Update ===" << "\n";
        out << "Updated: " + QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
        out << timestamp() + " [UPDATE] Metadata updated by: " + user << "\n";
        file.close();
    }
}

void WorkflowLogger::appendToLogWithChanges(const QString &folderPath, const QString &user, const QStringList &changes)
{
    QString appendLogPath = folderPath + "/.workflow.log";

    QFile file(appendLogPath);
    if (file.open(QIODevice::Append | QIODevice::Text)) {
        QTextStream out(&file);
        out << "\n";
        out << "=== Metadata Update ===" << "\n";
        out << "Updated: " + QDateTime::currentDateTime().toString(Qt::ISODate) << "\n";
        out << timestamp() + " [UPDATE] Metadata updated by: " + user << "\n";

        if (!changes.isEmpty()) {
            out << timestamp() + " [CHANGES]" << "\n";
            for (const QString &change : changes) {
                out << "         " + change << "\n";
            }
        } else {
            out << timestamp() + " [CHANGES] No changes detected" << "\n";
        }

        file.close();
    }
}

void WorkflowLogger::writeLog()
{
    if (logPath.isEmpty()) {
        return;
    }

    logEntries << "";
    logEntries << "Completed: " + QDateTime::currentDateTime().toString(Qt::ISODate);

    QFile file(logPath);
    if (file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QTextStream out(&file);
        for (const QString &entry : logEntries) {
            out << entry << "\n";
        }
        file.close();
    }
}
