#include "thumbnailgenerator.h"
#include <QProcess>
#include <QFile>
#include <QDir>
#include <QFileInfo>
#include <QStandardPaths>

ThumbnailGenerator::ThumbnailGenerator(QObject *parent)
    : QObject(parent)
{
}

QString ThumbnailGenerator::generateThumbnail(const QString &pdfPath, const QString &outputPath)
{
    QFile::remove(outputPath);

    QProcess process;
    QStringList args;
    args << "-png" << "-f" << "1" << "-l" << "1" << "-scale-to" << "1000"
         << "-singlefile" << pdfPath << outputPath.chopped(4);

    process.start("pdftoppm", args);
    process.waitForFinished(10000);

    if (process.exitCode() == 0 && QFile::exists(outputPath)) {
        return outputPath;
    }

    return QString();
}

QString ThumbnailGenerator::createThumbnail(const QString &pdfPath)
{
    QString cleanPath = pdfPath;
    if (cleanPath.startsWith("file://")) {
        cleanPath = cleanPath.mid(7);
    }

    if (!QFile::exists(cleanPath) || !cleanPath.toLower().endsWith(".pdf")) {
        return QString();
    }

    QFileInfo fi(cleanPath);
    originalBaseName = fi.completeBaseName();

    QString tempDir = QStandardPaths::writableLocation(QStandardPaths::TempLocation);
    currentThumbnailPath = tempDir + "/" + originalBaseName + ".png";

    QString result = generateThumbnail(cleanPath, currentThumbnailPath);
    if (result.isEmpty()) {
        currentThumbnailPath.clear();
        originalBaseName.clear();
    }

    return currentThumbnailPath;
}

QString ThumbnailGenerator::createStampedThumbnail(const QString &stampedPdfPath)
{
    QString cleanPath = stampedPdfPath;
    if (cleanPath.startsWith("file://")) {
        cleanPath = cleanPath.mid(7);
    }

    if (!QFile::exists(cleanPath) || !cleanPath.toLower().endsWith(".pdf")) {
        return QString();
    }

    QFileInfo fi(cleanPath);
    stampedBaseName = fi.completeBaseName();

    QString tempDir = QStandardPaths::writableLocation(QStandardPaths::TempLocation);
    currentStampedThumbnailPath = tempDir + "/" + stampedBaseName + ".png";

    QString result = generateThumbnail(cleanPath, currentStampedThumbnailPath);
    if (result.isEmpty()) {
        currentStampedThumbnailPath.clear();
        stampedBaseName.clear();
    }

    return currentStampedThumbnailPath;
}

bool ThumbnailGenerator::moveThumbnailsToFolder(const QString &targetFolder)
{
    QString thumbDir = targetFolder + "/.thumbnails";
    QDir dir;
    if (!dir.mkpath(thumbDir)) {
        return false;
    }

    bool success = true;

    if (!currentThumbnailPath.isEmpty() && QFile::exists(currentThumbnailPath)) {
        QString targetPath = thumbDir + "/" + originalBaseName + ".png";
        QFile::remove(targetPath);
        if (!QFile::rename(currentThumbnailPath, targetPath)) {
            if (QFile::copy(currentThumbnailPath, targetPath)) {
                QFile::remove(currentThumbnailPath);
            } else {
                success = false;
            }
        }
        currentThumbnailPath.clear();
    }

    if (!currentStampedThumbnailPath.isEmpty() && QFile::exists(currentStampedThumbnailPath)) {
        QString targetPath = thumbDir + "/" + stampedBaseName + ".png";
        QFile::remove(targetPath);
        if (!QFile::rename(currentStampedThumbnailPath, targetPath)) {
            if (QFile::copy(currentStampedThumbnailPath, targetPath)) {
                QFile::remove(currentStampedThumbnailPath);
            } else {
                success = false;
            }
        }
        currentStampedThumbnailPath.clear();
    }

    return success;
}

QString ThumbnailGenerator::getThumbnailPath() const
{
    return currentThumbnailPath;
}
