/*
 * XRechnung Viewer Suite
 *
 * Copyright (C) 2023 Klaas Freitag <opensource@freisturz.de>
 * Copyright (C) 2025 Stadt Treuchtlingen <sebastian.mogl@treuchtlingen.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef XRECHNUNG_H
#define XRECHNUNG_H

#include <QObject>
#include <QDomDocument>
#include <QUrl>
#include <QProcess>
#include <QTemporaryFile>

class XRechnung : public QObject
{
    Q_OBJECT
public:

	/// Enumerate possible serialization formats
	enum XRechnungFormat {
		UNKNOWN,
		UBL,
		CII
	};
	
    explicit XRechnung(QObject *parent = nullptr);

    XRechnung(const QUrl& file, QObject *parent = nullptr);
    QDomDocument domDocument() const;

    QString html() const;
    QUrl url() const;

    QString value(const QString& name) const;
    QString fileName() const { return _url.toLocalFile(); }
    QString intermediateFileName() const{ return _tempfile.fileName(); }

    void exportPdf(const QString& path);

    // config keys as static members
    static QString SaxonJar;
    static QString SaxonCii;
    static QString SaxonUbl;
    static QString SaxonHtml;
    static QString SaxonPdf;
    static QString SaxonXsl;
signals:
    void htmlAvailable();
    void domDocAvailable();

private slots:
    bool readDomDocument();
    void createIntermediate();
    void detectFormat();
    void slotErrorOccurred(QProcess::ProcessError error);
    void slotDomFinished(int exitCode, QProcess::ExitStatus exitStatus);
    void createHtml();
    void slotHtmlFinished(int exitCode, QProcess::ExitStatus exitStatus);
    void slotPdfFopFinished(int exitCode, QProcess::ExitStatus exitStatus);
    void slotPdfXslFinished(int exitCode, QProcess::ExitStatus exitStatus);

private:
    QUrl            _url;
    QString         _html;
    QDomDocument    _domDoc;
    QTemporaryFile  _tempfile;
    XRechnungFormat _format;
};

#endif // XRECHNUNG_H
